#include "cmd_base.h"
#include "lz4.h"

#include <util/generic/algorithm.h>
#include <util/generic/singleton.h>
#include <util/generic/utility.h>
#include <util/stream/format.h>
#include <util/folder/path.h>
#include <util/datetime/cputimer.h>
#include <library/cpp/json/json_writer.h>
#include <library/cpp/json/json_reader.h>

namespace {

template<class It>
std::pair<It, It> EqualPrefixRange(It it1,
                                   It it2,
                                   const TString &s)
{
    size_t sSize = s.size();

    auto lower = LowerBound(it1, it2, s, [sSize](const auto &f, const TString &s) {
        return f.Name.compare(0, sSize, s) < 0;
    });
    auto upper = UpperBound(it1, it2, s, [sSize](const TString &s, const auto &f) {
        return f.Name.compare(0, sSize, s) > 0;
    });
    return std::make_pair(lower, upper);
}

template<class It, class T, class Compare>
It BinaryFind(It first,
              It last,
              const T &value,
              Compare comp)
{
    It it;
    typename std::iterator_traits<It>::difference_type count, step;
    count = std::distance(first, last);

    while (count > 0) {
        it = first;
        step = count / 2;
        std::advance(it, step);
        if (comp(*it, value) < 0) {
            first = ++it;
            count -= step + 1;
        }
        else
            count = step;
    }
    return (first != last && !comp(*first, value)) ? first : last;
}

auto BinaryFindFileInfo(const TVector<NSolomon::TFileInfo>& kikimrFiles,
                        const IS3Client::TFileInfo& s3File,
                        const TString kikimrPrefix,
                        size_t s3PrefixLength,
                        bool checkFileSizes = true)
{
    IS3Client::TFileInfo shortS3File = {kikimrPrefix + s3File.Name.substr(s3PrefixLength + 1), s3File.SizeBytes, s3File.LastModified};

    auto it = BinaryFind(kikimrFiles.begin(), kikimrFiles.end(), shortS3File,
                        [checkFileSizes](const auto &v, const auto &e) {
                            int n = v.Name.compare(e.Name);
                            return ((n == 0 && checkFileSizes) ? (v.SizeBytes != e.SizeBytes) : n);
                        });
    return it;
}

TString Hostname() {
    TString fqdn;
    constexpr size_t maxNameSize = 1024;

    fqdn.ReserveAndResize(maxNameSize);
    if (gethostname(fqdn.begin(), maxNameSize) < 0) {
        ythrow yexception() << "Cannot gethostname: " << LastSystemErrorText();
    }
    fqdn.remove(fqdn.find('\0'));

    return fqdn;
}

bool PidExists(pid_t pid) {
    TFsPath pidDir = Sprintf("/proc/%u", pid);

    return pidDir.Exists();
}


/**
 * Dump KV-tablet files to S3.
 */
class TCmd3SDump: public TCliS3CommandTablet {
private:
    const TString LockFile          = ".dumplock";
    const TString StateFile         = ".state";
    const TString SnapshotPrefix    = "ks3bak.";
    const TString PrefixToBackup    = "c.";
    const TDuration ExpirationTime  = TDuration::Seconds(120);
    const size_t ReadChunkSize      = 16*1024*1024;

    TString         FQDN    = Hostname();
    pid_t           PID     = getpid();
    TSimpleTimer    LockTimer;
    bool            LockInited = false;

    bool ReadLock(ui64 tabletId,
                  TString &fqdn,
                  pid_t &pid,
                  ui32 &timestamp)
    {
        NJson::TJsonValue json;
        TString data = WaitAndCheck(KvClient_->ReadFile(tabletId, LockFile));

        if (!NJson::ReadJsonTree(data, &json)) {
            return false;
        }
        fqdn        = json["fqdn"].GetString();
        pid         = json["pid"].GetInteger();
        timestamp   = json["expireAt"].GetInteger();

        return true;
    }

    void WriteLock(ui64 tabletId) {
        NJson::TJsonValue json;

        json.InsertValue("fqdn", NJson::TJsonValue(FQDN));
        json.InsertValue("pid", NJson::TJsonValue(PID));
        json.InsertValue("expireAt", NJson::TJsonValue(TInstant::Now().Seconds() + ExpirationTime.Seconds()));

        WaitAndCheck(KvClient_->WriteFile(tabletId, LockFile, NJson::WriteJson(json)));
        LockTimer.Reset();
    }

    bool TryLockTablet(ui64 tabletId) {
        TString fqdn;
        ui32    timestamp;
        pid_t   pid         = 0;
        ui32    expire      = 0;

        if (LockInited && LockTimer.Get() < ExpirationTime/2) {
            return true;
        }
        timestamp = TInstant::Now().Seconds();
        if (ReadLock(tabletId, fqdn, pid, expire) && expire > timestamp) {
            if (fqdn != FQDN) {
                Cout << "Tablet " << tabletId
                     << " is locked from host " << fqdn
                     << " (pid=" << pid
                     << ") will expire in " << expire - timestamp << "s"
                     << Endl;
                return false;
            }
            if (pid != PID && PidExists(pid)) {
                Cout << "Tablet " << tabletId
                     << " is locked from this host (pid=" << pid
                     << ") will expire in " << expire - timestamp << "s"
                     << Endl;
                return false;
            }
        }
        WriteLock(tabletId);
        LockInited = true;
        return true;
    }

    void UnlockTablet(ui64 tabletId) {
        WaitAndCheck(KvClient_->RemoveFile(tabletId, LockFile));
    }

private:
    void Options(NLastGetopt::TOpts *opts) override {
        TCliS3CommandTablet::Options(opts);

        opts->AddLongOption("prefix")
                .Help("prefix for storage paths in s3")
                .RequiredArgument("PREFIX")
                .Required();
        opts->AddLongOption("dst-dir")
                .Help("destination directory in PREFIX, instead of shardId")
                .RequiredArgument("DIRECTORY")
                .DefaultValue("")
                .Optional();
        opts->AddLongOption("force")
                .Help("overwrite any file already in s3")
                .NoArgument()
                .Optional();
        opts->AddLongOption("amp")
                .Help("amplify upload by this factor")
                .RequiredArgument("FACTOR")
                .DefaultValue("1")
                .Optional();
        opts->AddLongOption("fast")
                .Help("use fast compression")
                .NoArgument()
                .Optional();
        opts->AddLongOption("hc")
                .Help("use high compression")
                .NoArgument()
                .Optional();
        opts->AddLongOption("safety-factor")
                .Help("fail if FACTOR*size(existing backup) > size(new backup)")
                .RequiredArgument("FACTOR")
                .DefaultValue("0.1")
                .Optional();
        opts->AddLongOption("stats")
                .Help("print statistics")
                .NoArgument()
                .Optional();
    }

    int RunS3OnTablet(ui64 tabletId, ui64 shardId, const NLastGetopt::TOptsParseResult &opts) override {
        ui32 counter = 0;
        std::stringstream *pss;
        std::unordered_map<std::string, std::stringstream>keyToFilestream;
        size_t bytes;
        ui32 totalFiles;
        ui64 bytesTotal     = 0;
        ui64 s3bytesTotal   = 0;
        ui64 bytesUpload    = 0;
        ui64 bytesUploaded  = 0;
        ui64 bytesRemove    = 0;
        int retryCount      = 3;
        TVector<NSolomon::TFileInfo> kikimrFilesToBackup;
        TVector<IS3Client::TFileInfo> s3Files;
        TVector<IS3Client::TDirInfo> s3Dirs;
        TVector<TString> s3FilesToRemove;
        TString s3Prefix    = TFsPath(opts.Get("prefix")).Fix().GetPath();
        TString dstDir      = opts.Get("dst-dir");
        bool forceOverwrite = opts.Has("force");
        bool compressFast   = opts.Has("fast");
        bool compressHigh   = opts.Has("hc");
        bool printStats     = opts.Has("stats");
        ui32 ampFactor      = std::stoul(opts.Get("amp"));
        double satefyFactor = std::stod(opts.Get("safety-factor"));
        TString s3StateFile;

        s3Prefix += (s3Prefix.EndsWith('/') ? "" : "/") + ((dstDir.size() > 0) ? dstDir : Sprintf("%04lu", shardId));
        s3StateFile = s3Prefix + "/" + StateFile;

        if (compressFast && compressHigh) {
            Cout << "Cannot use both High and Fast compression. Exiting ..." << Endl;
            return 1;
        }

        // lock tablet
        if (!TryLockTablet(tabletId)) {
            Cout << "Cannot lock tablet " << tabletId << ". Exiting ..." << Endl;
            return 1;
        }

        // if exist remove files with "SnapshotPrefix"
        Cout << "Removing old backup files ..." << Endl;
        WaitAndCheck(KvClient_->RemoveFiles(
            tabletId,
            {SnapshotPrefix + " ", false, SnapshotPrefix + "~", false}));

        // create snapshot
        Cout << "Creating new snapshot ..." << Endl;
        WaitAndCheck(KvClient_->CopyFiles(
            tabletId,
            {PrefixToBackup + " ", false, PrefixToBackup + "~", false},
            {},
            SnapshotPrefix));

        // list files
        kikimrFilesToBackup = WaitAndCheck(KvClient_->ListFiles(tabletId));
        SortBy(kikimrFilesToBackup, [](const auto &f) { return f.Name; });

        // remove from backup list all but "SnapshotPrefix" files
        auto p = EqualPrefixRange(kikimrFilesToBackup.begin(), kikimrFilesToBackup.end(), SnapshotPrefix);
        kikimrFilesToBackup.erase(p.second, kikimrFilesToBackup.end());
        kikimrFilesToBackup.erase(kikimrFilesToBackup.begin(), p.first);
        Cout << "Found " << std::distance(p.first, p.second) << " files with '" << SnapshotPrefix << "' prefix" << Endl;

        // s3 files at prefix
        if (!S3Client_->List(s3Files, s3Dirs, s3Prefix)) {
            Cout << "Failed to list S3 files at " << s3Prefix << Endl;
            UnlockTablet(tabletId);
            return 1;
        }
        SortBy(s3Files, [](const IS3Client::TFileInfo &f) { return f.Name; });

        for (const IS3Client::TFileInfo &f: s3Files) {
            s3bytesTotal += f.SizeBytes;
        }
        for (const auto& f: kikimrFilesToBackup) {
            bytesTotal += f.SizeBytes;
        }
        // find excess files at s3, remove from vector already backed up files, based on name (and size if not compressing)
        // bool compareBySize = (!compressFast && !compressHigh);
        // do not compare by size, since files in s3 could be already compressed
        bool compareBySize = false;
        for (const IS3Client::TFileInfo &f: s3Files) {
            auto it = BinaryFindFileInfo(kikimrFilesToBackup, f, SnapshotPrefix, s3Prefix.size(), compareBySize);
            if (it != kikimrFilesToBackup.end()) {
                ++counter;
                if (forceOverwrite) {
                    s3FilesToRemove.push_back(TString(f.Name.c_str()));
                    bytesRemove += f.SizeBytes;
                } else {
                    kikimrFilesToBackup.erase(it);
                }
            } else if (f.Name.compare(s3StateFile)) {
                s3FilesToRemove.push_back(TString(f.Name.c_str()));
                bytesRemove += f.SizeBytes;
            }
        }
        Cout << "Working at s3 prefix='"    << s3Prefix << "'. "
             << counter                     << " files already in s3 backup"
             << ((forceOverwrite) ? " (will be overwritten)" : "") << ". "
             << kikimrFilesToBackup.size()  << " files are to be backed up. "
             << s3FilesToRemove.size()      << " extra files in s3 will be removed."
             << Endl;
        if (printStats) {
            for (const auto& f: kikimrFilesToBackup) {
                bytesUpload += f.SizeBytes;
            }
            Cout << "STATS ON TABLET "              << tabletId
                 << " BYTES_TOTAL: "                << bytesTotal
                 << " BYTES_UPLOAD_UNCOMPRESSED: "  << bytesUpload
                 << " BYTES_REMOVE: "               << bytesRemove
                 << Endl;
        }
        if (satefyFactor >= 0 && satefyFactor*s3bytesTotal > bytesTotal) {
            Cout << "Fail to create backup since "
                 << "safety_factor*size(existing backup) > size(new backup): "
                 << Sprintf("%.3f*%.3f MB (= %.3f MB) > %.3f MB",
                            satefyFactor,
                            static_cast<double>(s3bytesTotal)/1024/1024,
                            satefyFactor*s3bytesTotal/1024/1024,
                            static_cast<double>(bytesTotal)/1024/1024
                            )
                 << Endl;
            return 1;
        }

        // upload state file
        pss = &keyToFilestream[s3StateFile];
        *pss << "# " << FQDN << "\n";
        for (const TString &f: s3FilesToRemove) {
            *pss << "- " << f << "\n";
        }
        for (const auto& f: kikimrFilesToBackup) {
            *pss << "+ " << s3Prefix + "/" + f.Name.substr(SnapshotPrefix.size()) << "\n";
        }
        pss->sync();
        if (!S3Client_->Put(keyToFilestream, bytes, retryCount)) {
            Cout << "\nFailed to upload update plan to s3://" << s3StateFile
                 << Endl;
            UnlockTablet(tabletId);
            return 1;
        }

        // upload
        counter = 0;
        totalFiles = kikimrFilesToBackup.size();
        auto it = kikimrFilesToBackup.begin();
        while (it != kikimrFilesToBackup.end()) {
            TSimpleTimer    simpleTimer;
            size_t          sizeOrig = 0;
            size_t          size = 0;
            uint32_t        errorCounter = 0;
            ui32            fileCounter;
            ui64            timeGetUs;
            ui64            timePutUs;
            TMap<TString, size_t> nameToSizes;
            TMap<TString, TVector<NSolomon::TAsyncKvResult<TString>>> nameToReads;

            if (!TryLockTablet(tabletId)) {
                Cout << "Cannot lock tablet " << tabletId << ". Exiting ..." << Endl;
                return 1;
            }
            simpleTimer.Reset();
            for (fileCounter = 0; fileCounter < ampFactor && it != kikimrFilesToBackup.end(); ++fileCounter, ++it, ++counter) {
                TVector<NSolomon::TAsyncKvResult<TString>> chunkToReads;
                TString origName = it->Name.substr(SnapshotPrefix.size());

                for (size_t offt = 0; offt < it->SizeBytes; offt += ReadChunkSize) {
                    chunkToReads.push_back(KvClient_->ReadFile(tabletId, it->Name, offt, ReadChunkSize));
                }
                nameToReads[origName] = chunkToReads;
                nameToSizes[origName] = it->SizeBytes;
                Cout << "Reading from kikimr " << tabletId << "/" << it->Name << " (size=" << it->SizeBytes << " chunks=" << chunkToReads.size() << ")." << Endl;
            }
            keyToFilestream.clear();
            for (auto &[ n, rs ]: nameToReads) {
                TString s3Name = s3Prefix + "/" + n;
                TString data = "";

                for (auto &r: rs) {
                    data += WaitAndCheck(r);
                }
                if (compressFast) {
                    TLz4Compress zFile(&keyToFilestream[s3Name]);
                    zFile.UseDefault();
                    zFile << data;
                    // zFile.Flush();
                } else if (compressHigh) {
                    TLz4Compress zFile(&keyToFilestream[s3Name]);
                    zFile.UseHC();
                    zFile << data;
                    // zFile.Flush();
                } else {
                    keyToFilestream[s3Name] << data;
                    // keyToFilestream[n].flush();
                }
                if (data.size() != nameToSizes[n]) {
                    Cout << "Got bad file from kv " << n << ", size=" << data.size() << ", want=" << nameToSizes[n] << Endl;
                    errorCounter++;
                }
                sizeOrig += data.size();
                size += keyToFilestream[s3Name].str().size();
            }
            timeGetUs = simpleTimer.Get().MicroSeconds();
            Cout << Sprintf("Read %u files from kikimr, %.3f MB/s on total of %.3f MB (compress ratio %.3f -> %.3f MB). Sending them to s3",
                            fileCounter,
                            static_cast<double>(sizeOrig)/static_cast<double>(timeGetUs),
                            static_cast<double>(sizeOrig)/1024/1024,
                            static_cast<double>(sizeOrig)/static_cast<double>(size),
                            static_cast<double>(size)/1024/1024) << Endl;
            size = 0;
            simpleTimer.Reset();
            if (!S3Client_->Put(keyToFilestream, size, retryCount)) {
                Cout << "\nFailed to upload " << fileCounter << " files " << " to s3://" << s3Prefix << Endl;
                UnlockTablet(tabletId);
                return 1;
            }
            timePutUs = simpleTimer.Get().MicroSeconds();
            Cout << Sprintf("Uploaded %u files to s3 (%u/%u), %.3f MB/s on total of %.3f MB.",
                            fileCounter,
                            counter,
                            totalFiles,
                            static_cast<double>(size)/static_cast<double>(timePutUs),
                            static_cast<double>(size)/1024/1024) << Endl;
            if (printStats) {
                bytesUploaded += size;
                bytesUpload -= sizeOrig;
                Cout << "STATS ON DUMP"
                     << " TOTAL_BYTES_SENT: "   << bytesUploaded
                     << " TOTAL_BYTES_LEFT: "   << bytesUpload
                     << " CHUNK_SIZE: "         << size
                     << " TIME_GET_US: "        << timeGetUs
                     << " TIME_PUT_US: "        << timePutUs
                     << " ERRORS: "             << errorCounter
                     << Endl;
            }
        }

        if (!TryLockTablet(tabletId)) {
            Cout << "Cannot lock tablet " << tabletId << ". Exiting ..." << Endl;
            return 1;
        }

        // remove from s3 excess files
        if (s3FilesToRemove.size() > 0) {
            Cout << "Removing extra files from s3 ..." << Endl;
            for (const TString &f: s3FilesToRemove) {
                Cout << " del " << f << Endl;
            }
            if (!S3Client_->Del(s3FilesToRemove)) {
                Cout << "Failed to delete extra S3 files." << Endl;
            }
        }

        // on success, remove snapshot
        Cout << "Backup is done. Removing snapshot ..." << Endl;
        WaitAndCheck(KvClient_->RemoveFiles(tabletId, {SnapshotPrefix + " ", false, SnapshotPrefix + "~", false}));

        // remove lock
        Cout << "Backup is done. Removing table lock ..." << Endl;
        UnlockTablet(tabletId);

        // remove state file
        if (!S3Client_->Del({s3StateFile})) {
            Cout << "Failed to delete state file from S3." << Endl;
        }

        return 0;
    }
};

} // namespace

TMainClass* Cmd3SDump() {
    return Singleton<TCmd3SDump>();
}
