#include "cmd_base.h"

#include <util/generic/algorithm.h>
#include <util/generic/singleton.h>
#include <util/generic/utility.h>
#include <util/stream/format.h>

#include <cmath>

using NLastGetopt::TOptsParseResult;

namespace {

constexpr TStringBuf COLUMN_NAME = "NAME";
constexpr TStringBuf COLUMN_SIZE = "SIZE";
constexpr TStringBuf COLUMN_TIME = "TIME";

template <typename T>
size_t WidthOf(T v) {
    if (v == 0) {
        return 1;
    }
    size_t add = (v > 0) ? 1 : 2; // minus sign
    return add + static_cast<size_t>(std::log10(static_cast<double>(v)));
}

/**
 * List s3 files.
 */
class TCmdS3Ls: public TCliS3CommandBase {
private:
    void Options(NLastGetopt::TOpts* opts) override {
        opts->AddLongOption("prefix")
                .Help("prefix of s3 paths")
                .RequiredArgument("PREFIX")
                .DefaultValue("")
                .Optional();
        opts->AddLongOption("sort")
                .Help("sort files by specified field {NAME,SIZE,TIME}, add '-' "
                      "in front to sort in reverse order")
                .RequiredArgument("FIELD")
                .Optional();
        opts->AddLongOption("max-keys")
                .Help("maximum number of files to list")
                .RequiredArgument("NUMBER")
                .DefaultValue("10000")
                .Optional();
        opts->AddLongOption("recursive")
                .Help("do recursive files listing")
                .NoArgument()
                .Optional();
    }

    int Run(const TOptsParseResult& opts) override {
        bool recursive  = opts.Has("recursive");
        TString prefix  = opts.Get("prefix");
        ui64 maxKeys    = std::stoull(opts.Get("max-keys"));

        TVector<IS3Client::TFileInfo> files;
        TVector<IS3Client::TDirInfo> dirs;

        if (recursive) {
            if (!S3Client_->ListRecursive(files, prefix, maxKeys))
                return 1;
        } else {
            if (!S3Client_->List(files, dirs, prefix, maxKeys))
                return 1;
        }

        if (files.empty() && dirs.empty()) {
            Cerr << "no files or dirs here" << Endl;
            return 0;
        }

        TStringBuf sort = opts.GetOrElse("sort", "");
        bool reverseOrder = false;
        if (sort && sort[0] == '-') {
            sort = sort.Skip(1);
            reverseOrder = true;
        }
        SortBy(dirs, [](const IS3Client::TDirInfo& f) { return f.Name; });

        if (sort == TStringBuf("NAME")) {
            SortBy(files, [](const IS3Client::TFileInfo& f) { return f.Name; });
        } else if (sort == TStringBuf("SIZE")) {
            SortBy(files, [](const IS3Client::TFileInfo& f) { return f.SizeBytes; });
        } else if (sort == TStringBuf("TIME")) {
            SortBy(files, [](const IS3Client::TFileInfo& f) { return f.LastModified; });
        }

        size_t nameWidth = COLUMN_NAME.size();
        ui64 maxSize = 0;
        for (const IS3Client::TDirInfo& dir: dirs) {
            nameWidth = Max(nameWidth, dir.Name.size());
        }
        for (const IS3Client::TFileInfo& file: files) {
            nameWidth = Max(nameWidth, file.Name.size());
            maxSize = Max(maxSize, file.SizeBytes);
        }
        nameWidth += 2;
        size_t sizeWidth = Max(WidthOf(maxSize), COLUMN_SIZE.size()) + 2;

        Cout << RightPad(COLUMN_NAME, nameWidth)
             << RightPad(COLUMN_SIZE, sizeWidth)
             << COLUMN_TIME
             << '\n';

        if (reverseOrder) {
            for (auto it = dirs.rbegin(), end = dirs.rend(); it != end; ++it) {
                PrintDir(*it, nameWidth);
            }
            for (auto it = files.rbegin(), end = files.rend(); it != end; ++it) {
                PrintFile(*it, nameWidth, sizeWidth);
            }
        } else {
            for (const IS3Client::TDirInfo& dir: dirs) {
                PrintDir(dir, nameWidth);
            }
            for (const IS3Client::TFileInfo& file: files) {
                PrintFile(file, nameWidth, sizeWidth);
            }
        }

        Cout << Flush;
        return 0;
    }

    void PrintDir(const IS3Client::TDirInfo& dir, size_t nameWidth) {
        Cout << RightPad(dir.Name, nameWidth)
             << "<dir>\n";
    }

    void PrintFile(const IS3Client::TFileInfo& file, size_t nameWidth, size_t sizeWidth) {
        Cout << RightPad(file.Name, nameWidth)
             << RightPad(file.SizeBytes, sizeWidth)
             << file.LastModified.ToStringUpToSeconds()
             << '\n';
    }
};

} // namespace

TMainClass* CmdS3Ls() {
    return Singleton<TCmdS3Ls>();
}
