#include "cmd_base.h"
#include "lz4.h"

#include <util/generic/singleton.h>
#include <util/datetime/cputimer.h>
#include <util/stream/file.h>
#include <fstream>
#include <iostream>

namespace {

/**
 * Read file from s3.
 */
class TCmdS3Read: public TCliS3CommandBase {
private:
    void Options(NLastGetopt::TOpts* opts) override {
        opts->AddLongOption('r', "retry")
                .Help("retry count")
                .RequiredArgument("COUNT")
                .DefaultValue("2")
                .Optional();
        opts->AddLongOption('s', "skip-decompress")
                .Help("skip decompression of s3 file")
                .NoArgument()
                .Optional();
        opts->SetFreeArgsNum(1, 2);
        opts->SetFreeArgTitle(0, "src_file", "name of the file to read from");
        opts->SetFreeArgTitle(1, "dst_file", "name of the file to write to, or skip for <stdout>");
    }

    int Run(const NLastGetopt::TOptsParseResult& opts) override {
        bool result;
        bool quiet = opts.Has("quiet");
        bool outFile = opts.GetFreeArgs().size() > 1;
        bool skipDecompress = opts.Has("skip-decompress");
        size_t bytes;
        TSimpleTimer timer;
        TString srcFilename = opts.GetFreeArgs()[0];
        int retryCount = std::stol(opts.Get("retry"));

        S3Client_->SetDebug(!quiet && outFile);

        std::unordered_map<std::string, std::stringstream> keyToStringStream;
        keyToStringStream[srcFilename] = std::stringstream(std::ios_base::binary|std::ios_base::out|std::ios_base::in);

        std::fstream dstFile;
        if (outFile) {
            TString dstFilename = opts.GetFreeArgs()[1];
            dstFile.open(dstFilename, std::ios_base::binary|std::ios_base::out|std::ios_base::trunc);
        } else if (opts.Has("retry")) {
            Cerr << "Cannot retry on stdout" << Endl;
            return 1;
        }
        std::iostream ios((outFile) ? dstFile.rdbuf() : std::cout.rdbuf());

        timer.Reset();
        result = S3Client_->Get(keyToStringStream, bytes, retryCount);

        if (skipDecompress) {
            ios << keyToStringStream[srcFilename].str();
        } else {
            try {
                TLz4Decompress zFile(&keyToStringStream[srcFilename]);
                zFile.ReadAll(ios);
            } catch (TLZ4TypeError) {
                ios << keyToStringStream[srcFilename].str();
            }
        }

        if (!quiet) {
            Cerr << "Got " << bytes/1024/1024 << " MB, "
                 << bytes/timer.Get().MicroSeconds()/1.024/1.024 << " MB/s"
                 << Endl;
        }
        return !result;
    }
};

} // namespace

TMainClass* CmdS3Read() {
    return Singleton<TCmdS3Read>();
}
