#include "cmd_base.h"
#include "lz4.h"

#include <util/generic/singleton.h>
#include <util/datetime/cputimer.h>
#include <util/stream/file.h>
#include <fstream>

namespace {

/**
 * Writes data into an s3 file.
 */
class TCmdS3Write: public TCliS3CommandBase {
private:
    void Options(NLastGetopt::TOpts* opts) override {
        opts->AddLongOption('i', "input")
                .Help("file to read data from")
                .RequiredArgument("FILE")
                .Optional();
        opts->AddLongOption('r', "retry")
                .Help("retry count")
                .RequiredArgument("COUNT")
                .DefaultValue("2")
                .Optional();
        opts->AddLongOption("fast")
                .Help("use fast compression")
                .NoArgument()
                .Optional();
        opts->AddLongOption("hc")
                .Help("use high compression")
                .NoArgument()
                .Optional();
        opts->SetFreeArgsNum(1);
        opts->SetFreeArgTitle(0, "filename", "name of the file to write to");
    }

    int Run(const NLastGetopt::TOptsParseResult& opts) override {
        bool result;
        size_t bytes;
        TSimpleTimer timer;
        TString name = opts.GetFreeArgs()[0];
        bool compressFast = opts.Has("fast");
        bool compressHigh = opts.Has("hc");
        TString inFilename = (opts.Has("input")) ? opts.Get("input") : "";
        int retryCount = std::stol(opts.Get("retry"));
        std::unordered_map<std::string, std::stringstream>keyToFilestream;

        if (compressFast || compressHigh) {
            TLz4Compress zFile(&keyToFilestream[name]);
            if (compressFast) {
                zFile.UseDefault();
            } else {
                zFile.UseHC();
            }
            if (inFilename) {
                TFileInput srcFile(inFilename, std::ios_base::binary|std::ios_base::in);
                srcFile.ReadAll(zFile);
                zFile.Flush();
            } else {
                Cin.ReadAll(zFile);
                zFile.Flush();
            }
            result = S3Client_->Put(keyToFilestream, bytes, retryCount);
        } else {
            if (inFilename) {
                std::unordered_map<std::string, std::string>keyToFilename;
                keyToFilename[name] = inFilename;
                result = S3Client_->Put(keyToFilename, bytes, retryCount);
            } else {
                keyToFilestream[name] << std::cin.rdbuf();
                result = S3Client_->Put(keyToFilestream, bytes, retryCount);
            }
        }

        if (!opts.Has("quiet")) {
            Cout << "Sent " << bytes/1024/1024 << " MB, "
                 << bytes/timer.Get().MicroSeconds()/1.024/1.024 << " MB/s"
                 << Endl;
        }
        return !result;
    }
};

} // namespace

TMainClass* CmdS3Write() {
    return Singleton<TCmdS3Write>();
}
