#include "cmd_base.h"

#include <util/generic/singleton.h>
#include <util/stream/file.h>

namespace {

/**
 * Writes data into a file on specific KV-tablet associated with Stockpile shard.
 */
class TCmdWrite: public TCliCommandTablet {
private:
    void Options(NLastGetopt::TOpts* opts) override {
        TCliCommandTablet::Options(opts);
        opts->AddLongOption('i', "input")
                .Help("file to read data from")
                .RequiredArgument("FILE")
                .Optional();
        opts->SetFreeArgsNum(1);
        opts->SetFreeArgTitle(0, "filename", "name of the file to write to");
    }

    int RunOnTablet(ui64 tabletId, ui64, const NLastGetopt::TOptsParseResult& opts) override {
        TString filename = opts.GetFreeArgs()[0];
        TString chunkPrefix = "chunk.";
        size_t writeChunkSize = 16*1024*1024;

        TString data;
        if (opts.Has("input")) {
            TFileInput file(opts.Get("input"));
            data = file.ReadAll();
        } else {
            data = Cin.ReadAll();
        }

        if (data.size() > writeChunkSize) {
            TVector<TString> writeChunks;
            TVector<NSolomon::TAsyncKvResult<void>> writes;

            for (size_t offt = 0; offt < data.size(); offt += writeChunkSize) {
                TString chunkName = Sprintf("%s%s.%03lu", chunkPrefix.c_str(), filename.c_str(), writeChunks.size());
                writes.push_back(KvClient_->WriteFile(tabletId, chunkName, data.substr(offt, writeChunkSize)));
                writeChunks.push_back(chunkName);
            }
            for (auto &write: writes) {
                WaitAndCheck(write);
            }
            WaitAndCheck(KvClient_->ConcatFiles(tabletId, writeChunks, filename, false));
        } else {
            WaitAndCheck(KvClient_->WriteFile(tabletId, filename, data));
        }
        return 0;
    }
};

} // namespace

TMainClass* CmdWrite() {
    return Singleton<TCmdWrite>();
}
