package main

import (
	"crypto/tls"
	"crypto/x509"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"net"
	"net/http"
	"net/url"
	"os"
	"regexp"
	"strings"
	"time"

	"gopkg.in/yaml.v2"
)

// ==========================================================================================

var ConductorHostname = "c.yandex-team.ru"

var Config = ConfigStruct{
	ServiceAddr:    "[::]:8080",
	KubeConfigFile: "/root/.kube/config",
	KubeCluster:    "default",
	KubeUser:       "admin",
	CacheMaxTime:   Duration{5 * time.Minute},
	CacheBadTime:   Duration{10 * time.Second},
	ClientTimeout:  Duration{5 * time.Second},
	PeerTimeout:    Duration{2 * time.Second},
	Verbose:        true,
}

// ==========================================================================================

var ReRequiest = regexp.MustCompile("^/api/groups2hosts/[a-z-_]+$")
var MCache = NewMacroCache()
var RResolver = NewResolver(50, 5*time.Minute, 20*time.Second, false)

// ==========================================================================================

type KuberEndpointStruct struct {
	APIVersion string `json:"apiVersion"`
	Kind       string `json:"kind"`
	Metadata   struct {
		Name              string `json:"name"`
		NameSpace         string `json:"namespace"`
		ResourceVersion   string `json:"resourceVersion"`
		UID               string `json:"uid"`
		CreationTimestamp string `json:"creationTimestamp"`
	} `json:"metadata"`
	Subsets []struct {
		Addresses []struct {
			IP        string `json:"ip"`
			NodeName  string `json:"nodeName"`
			TargetRef struct {
				Kind            string `json:"kind"`
				Name            string `json:"name"`
				NameSpace       string `json:"namespace"`
				ResourceVersion string `json:"resourceVersion"`
				UID             string `json:"uid"`
			} `json:"targetRef"`
		} `json:"addresses"`
	} `json:"subsets"`
}

type KubeConfigStruct struct {
	APIVersion string `yaml:"apiVersion"`
	Kind       string `yaml:"kind"`
	Clusters   []struct {
		Name    string `yaml:"name"`
		Cluster struct {
			CertificateAuthorityData string `yaml:"certificate-authority-data"`
			Server                   string `yaml:"server"`
		} `yaml:"cluster"`
	} `yaml:"clusters"`
	Users []struct {
		Name string `yaml:"name"`
		User struct {
			ClientCertificateData string `yaml:"client-certificate-data"`
			ClientKeyData         string `yaml:"client-key-data"`
		} `yaml:"user"`
	} `yaml:"users"`
}

type KuberGroupStruct struct {
	NameSpace string `json:"namespace"`
	Service   string `json:"service"`
}

type ConfigStruct struct {
	ServiceAddr      string `json:"service_addr"`
	KubeConfigFile   string `json:"kube_config_file"`
	KubeConfig       KubeConfigStruct
	KubeCluster      string `json:"kube_cluster"`
	KubeUser         string `json:"kube_user"`
	KubeServer       string
	KubeTLSConfig    *tls.Config
	CacheMaxTime     Duration                    `json:"cache_max_time"`
	CacheBadTime     Duration                    `json:"cache_bad_time"`
	ClientTimeout    Duration                    `json:"client_timeout"`
	PeerTimeout      Duration                    `json:"peer_timeout"`
	ConductorToKuber map[string]KuberGroupStruct `json:"conductor_to_kuber"`
	Verbose          bool                        `json:"verbose"`
}

// ==========================================================================================

func readConfig(path string) {
	data, err := ioutil.ReadFile(path)
	if err != nil {
		log.Fatalf("Bad config file path %s: %s", path, err.Error())
	}
	if err = json.Unmarshal(data, &Config); err != nil {
		log.Fatalf("Bad json in config file %s: %s", path, err.Error())
	}
}

func readKubeConfig(path string, kubeConfig *KubeConfigStruct) {
	if path != "" {
		data, err := ioutil.ReadFile(path)
		if err != nil {
			log.Printf("Bad config file path %s: %s", path, err.Error())
		}
		if yaml.Unmarshal(data, kubeConfig) != nil {
			log.Fatalf("Bad yaml in config file %s", path)
		}
		if kubeConfig.Kind != "Config" {
			log.Fatalf("Bad kube config: kind is '%s' and not 'Config'", kubeConfig.Kind)
		}
	}
}

// ==========================================================================================

func conductor(g string) ([]string, error) {
	result := []string{}

	client := &http.Client{Timeout: Config.PeerTimeout.Duration}
	urlString := fmt.Sprintf("http://%s/api/groups2hosts/%s", ConductorHostname, g)

	resp, err := client.Get(urlString)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	body, err := io.ReadAll(resp.Body)
	if err != nil {
		return nil, fmt.Errorf("failed read response body, %v", err)
	}
	if resp.StatusCode == http.StatusNotFound {
		return result, fmt.Errorf("not found (404), %s", string(body))
	}
	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("bad response (%d), %s", resp.StatusCode, string(body))
	}

	names := strings.Split(strings.Trim(string(body), " \n"), "\n")
	for _, addrs := range RResolver.Resolv(names) {
		result = append(result, addrs...)
	}
	return result, nil
}

func kuber(nameSpace string, service string) ([]string, error) {
	result := []string{}
	ep := KuberEndpointStruct{}

	client := &http.Client{
		Transport: &http.Transport{TLSClientConfig: Config.KubeTLSConfig},
		Timeout:   Config.PeerTimeout.Duration,
	}
	urlString := fmt.Sprintf("%s/api/v1/namespaces/%s/endpoints/%s-service?limit=4096", Config.KubeServer, nameSpace, service)
	u, err := url.Parse(urlString)
	if err != nil {
		log.Fatalf("Failed to parse url (%s): %v", urlString, err)
	}
	req := http.Request{
		URL:    u,
		Header: http.Header{"Accept": []string{"application/json"}},
		Close:  true,
	}
	resp, err := client.Do(&req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	body, err := io.ReadAll(resp.Body)
	if err != nil {
		return nil, fmt.Errorf("failed read response body, %v", err)
	}
	if resp.StatusCode == http.StatusNotFound {
		return result, fmt.Errorf("not found (404), %s", string(body))
	}
	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("bad response (%d), %s", resp.StatusCode, string(body))
	}

	if err = json.Unmarshal(body, &ep); err != nil {
		return nil, fmt.Errorf("bad response for %s: %v", urlString, err)
	}
	for _, subset := range ep.Subsets {
		for _, addresses := range subset.Addresses {
			result = append(result, addresses.IP)
		}
	}
	return result, nil
}

// ==========================================================================================

type ResolverRequest struct {
	Req   string
	Resp  []string
	Error error
	Cache *Cache
}

type Resolver struct {
	Cache   *Cache
	Factory *Factory
}

func NewResolver(nWorkers uint, goodTime, badTime time.Duration, debug bool) *Resolver {
	getter := func(r interface{}) (interface{}, error) {
		req := r.(string)
		if net.ParseIP(req) != nil {
			names, err := net.LookupAddr(req)
			result := []string{}
			for _, name := range names {
				result = append(result, strings.TrimRight(name, "."))
			}
			return result, err
		} else {
			if !strings.HasSuffix(req, ".") {
				req = req + "."
			}
			return net.LookupHost(req)
		}
	}
	copier := func(value interface{}) interface{} {
		result := []string{}
		result = append(result, value.([]string)...)
		return result
	}
	resolver := func(r interface{}) {
		req := r.(*ResolverRequest)
		resp, err := req.Cache.Get(req.Req)
		req.Resp = resp.([]string)
		req.Error = err
	}
	r := &Resolver{
		Cache:   NewCache("resolver", getter, copier, goodTime, badTime, debug),
		Factory: NewFactory(nWorkers, resolver, debug),
	}
	return r
}

func (r *Resolver) Resolv(reqs []string) map[string][]string {
	intReq := make([]interface{}, 0, len(reqs))
	for _, req := range reqs {
		intReq = append(intReq, &ResolverRequest{
			Req:   req,
			Resp:  []string{},
			Cache: r.Cache,
		})
	}
	r.Factory.Do(intReq)
	result := map[string][]string{}
	for _, req := range intReq {
		result[req.(*ResolverRequest).Req] = req.(*ResolverRequest).Resp
	}
	return result
}

// ==========================================================================================

type MacroCache struct {
	Cache *Cache
}

func NewMacroCache() *MacroCache {
	getter := func(key interface{}) (interface{}, error) {
		kResult := []string{}
		var err, kErr error

		if k, ok := Config.ConductorToKuber[key.(string)]; ok {
			if Config.Verbose {
				log.Printf("Sending request to kubernetes api (namespace=%s, service=%s)", k.NameSpace, k.Service)
			}
			kResult, kErr = kuber(k.NameSpace, k.Service)
			if kErr != nil && kResult == nil {
				return nil, fmt.Errorf("kuber api for %s returned: %v", key, kErr)
			}
		}
		if Config.Verbose {
			log.Printf("Sending request to conductor api (key=%s)", key)
		}
		result, err := conductor(key.(string))
		if err != nil {
			return result, fmt.Errorf("conductor api for %s returned: %v, %v", key, err, kErr)
		}
		result = append(result, kResult...)
		return result, nil
	}
	copier := func(value interface{}) interface{} {
		result := []string{}
		result = append(result, value.([]string)...)
		return result
	}

	return &MacroCache{
		Cache: NewCache("macro", getter, copier, Config.CacheMaxTime.Duration, Config.CacheBadTime.Duration, Config.Verbose),
	}
}

func (m *MacroCache) Get(key string) ([]string, error) {
	result, err := m.Cache.Get(key)
	return result.([]string), err
}

// ==========================================================================================

func reqHandler(w http.ResponseWriter, r *http.Request) {
	beginTime := time.Now()
	if Config.Verbose {
		log.Printf("Connection from %s: %s", r.RemoteAddr, r.URL.Path)
	}
	w.Header().Set("Content-type", "text/plain")
	w.Header().Set("Connection", "Close")
	if !ReRequiest.MatchString(r.URL.Path) {
		log.Printf("Unknown request 501 (%s): %v, url path=%s", r.RemoteAddr, time.Since(beginTime), r.URL.Path)
		w.WriteHeader(http.StatusNotImplemented)
		return
	}
	key := r.URL.Path[18:]
	result, err := MCache.Get(key)
	if err != nil {
		if result == nil {
			log.Printf("Failed to get result, 500 (%s, %s): %v, %v", r.RemoteAddr, key, time.Since(beginTime), err)
			w.WriteHeader(http.StatusInternalServerError)
		} else {
			log.Printf("Not found, 404 (%s, %s): %v, %v", r.RemoteAddr, key, time.Since(beginTime), err)
			w.WriteHeader(http.StatusNotFound)
		}
		return
	}
	log.Printf("Accepted request, 200 (%s, %s): %v, %v", r.RemoteAddr, key, time.Since(beginTime), result)
	w.WriteHeader(http.StatusOK)
	if _, err = fmt.Fprintf(w, "%s\n", strings.Join(result, "\n")); err != nil {
		log.Printf("Failed to send response (%s, %s): %v, %v", r.RemoteAddr, key, time.Since(beginTime), err)
	}
}

func serviceHTTP() {
	server := &http.Server{
		Addr:           Config.ServiceAddr,
		Handler:        http.HandlerFunc(reqHandler),
		ReadTimeout:    Config.ClientTimeout.Duration,
		WriteTimeout:   Config.ClientTimeout.Duration,
		MaxHeaderBytes: 1 << 14,
	}
	log.Fatal(server.ListenAndServe())
}

// ==========================================================================================

func main() {
	log.SetFlags(log.Ldate | log.Ltime | log.Lmicroseconds)
	if len(os.Args) != 2 {
		log.Fatalf("Usage: %s <config_file>", os.Args[0])
	}
	configFileName := os.Args[1]

	readConfig(configFileName)
	readKubeConfig(Config.KubeConfigFile, &Config.KubeConfig)

	var err error
	var certificateAuthority []byte
	var clientCertificate []byte
	var clientKey []byte
	for idx, user := range Config.KubeConfig.Users {
		if user.Name == Config.KubeUser {
			p := &Config.KubeConfig.Users[idx].User.ClientCertificateData
			if clientCertificate, err = base64.StdEncoding.DecodeString(*p); err != nil {
				log.Fatalf("Failed to decode client-certificate-data from %s: %v", Config.KubeConfigFile, err)
			}
			p = &Config.KubeConfig.Users[idx].User.ClientKeyData
			if clientKey, err = base64.StdEncoding.DecodeString(*p); err != nil {
				log.Fatalf("Failed to decode client-key-data from %s: %v", Config.KubeConfigFile, err)
			}
		}
	}
	for idx, cluster := range Config.KubeConfig.Clusters {
		if cluster.Name == Config.KubeCluster {
			p := &Config.KubeConfig.Clusters[idx].Cluster.CertificateAuthorityData
			if certificateAuthority, err = base64.StdEncoding.DecodeString(*p); err != nil {
				log.Fatalf("Failed to decode certificate-authority-data from %s: %v", Config.KubeConfigFile, err)
			}
			Config.KubeServer = Config.KubeConfig.Clusters[idx].Cluster.Server
		}
	}
	if len(clientCertificate) == 0 {
		log.Fatalf("Failed to get client-certificate from %s", Config.KubeConfigFile)
	}
	if len(clientKey) == 0 {
		log.Fatalf("Failed to get client-key from %s", Config.KubeConfigFile)
	}
	if len(certificateAuthority) == 0 {
		log.Fatalf("Failed to get certificate-authority from %s", Config.KubeConfigFile)
	}
	if len(Config.KubeServer) == 0 {
		log.Fatalf("Failed to get kuber server from %s", Config.KubeConfigFile)
	}

	cert, err := tls.X509KeyPair(clientCertificate, clientKey)
	if err != nil {
		log.Fatalf("Failed to set TLS certificate: %v", err)
	}
	rootCAs := x509.NewCertPool()
	if ok := rootCAs.AppendCertsFromPEM(certificateAuthority); !ok {
		log.Fatalf("Failed to set root CA certificate: %v", err)
	}
	Config.KubeTLSConfig = &tls.Config{
		Certificates:       []tls.Certificate{cert},
		RootCAs:            rootCAs,
		InsecureSkipVerify: false,
	}

	serviceHTTP()
}
