package main

import (
	"crypto/tls"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log"
	"net/http"
	"net/url"
	"sync"
	"sync/atomic"
	"time"
)

// ==========================================================================================

func getURL(urlString string,
	headers map[string]string,
	timeout time.Duration,
	TLSConfig *tls.Config,
	closeConnection bool) ([]byte, int, error) {

	client := &http.Client{
		Timeout: timeout,
	}
	if TLSConfig != nil {
		/*
			&tls.Config{
				Certificates: []tls.Certificate{cert},
				RootCAs: rootCAs,
				InsecureSkipVerify: false,
			}
		*/
		client.Transport = &http.Transport{TLSClientConfig: TLSConfig}
	}
	u, err := url.Parse(urlString)
	if err != nil {
		return nil, 0, err
	}
	h := make(http.Header)
	for k, v := range headers {
		h.Set(k, v)
	}
	req := http.Request{
		URL:    u,
		Header: h,
		Close:  closeConnection,
	}
	resp, err := client.Do(&req)
	if err != nil {
		return nil, 0, err
	}
	defer resp.Body.Close()
	body, err := io.ReadAll(resp.Body)
	if err != nil {
		return nil, resp.StatusCode, err
	}
	return body, resp.StatusCode, nil
}

// ==========================================================================================

// Do marshalling and unmarshalling of time for JSON.
// Time could be expressed as float64/Seconds or string/Go time format
//
type Duration struct {
	time.Duration
}

func (d Duration) MarshalJSON() ([]byte, error) {
	return json.Marshal(d.String())
}

func (d *Duration) UnmarshalJSON(b []byte) error {
	var v interface{}
	if err := json.Unmarshal(b, &v); err != nil {
		return err
	}
	switch value := v.(type) {
	case float64:
		d.Duration = time.Duration(value * float64(time.Second))
		return nil
	case string:
		var err error
		if d.Duration, err = time.ParseDuration(value); err != nil {
			return err
		}
		return nil
	default:
		return errors.New("invalid duration")
	}
}

// ==========================================================================================

// BgWorker is running in separate goroutine
//
type BgWorker struct {
	doneChan chan struct{}
	result   interface{}
	resErr   error
}

func NewBgWorker(input interface{}, worker func(interface{}) (interface{}, error)) *BgWorker {
	b := &BgWorker{
		doneChan: make(chan struct{}),
	}
	go func() {
		b.result, b.resErr = worker(input)
		close(b.doneChan)
	}()
	return b
}

func (b *BgWorker) Poll() (interface{}, bool, error) {
	select {
	case <-b.doneChan:
		return b.result, true, b.resErr
	default:
		return nil, false, nil
	}
}

func (b *BgWorker) Wait() (interface{}, error) {
	<-b.doneChan
	return b.result, b.resErr
}

// ==========================================================================================

// Cache that is transparently getting expired or absent values.
// If getting a value is delayed, then several Get calls line up in queue, not overloading
// the source of values.
//
// type Cache interface {
//     Get(key interface{}) (interface{}, error)
//     Delete(key interface{})
// }
//
type Cache struct {
	name          string
	getter        Getter
	copier        Copier
	GoodCacheTime time.Duration
	BadCacheTime  time.Duration
	mutex         sync.RWMutex
	data          map[interface{}]*CacheValue
	Debug         bool
	cleanUpTime   time.Duration
}

type CacheValue struct {
	Value       interface{}
	Error       error
	EndOfLife   time.Time
	WaitChannel chan struct{}
}

type Getter func(interface{}) (interface{}, error)
type Copier func(interface{}) interface{}

func NewCache(name string,
	getFunc Getter,
	copyFunc Copier,
	goodCacheTime time.Duration,
	badCacheTime time.Duration,
	debug bool) *Cache {

	c := &Cache{
		name:          name,
		getter:        getFunc,
		copier:        copyFunc,
		GoodCacheTime: goodCacheTime,
		BadCacheTime:  badCacheTime,
		data:          make(map[interface{}]*CacheValue),
		Debug:         debug,
		cleanUpTime:   5 * time.Second,
	}
	go c.cleanup()
	return c
}

func (c *Cache) cleanup() {
	for {
		c.mutex.Lock()
		t := time.Now()
		cacheLength := len(c.data)
		for k, v := range c.data {
			if !v.EndOfLife.IsZero() && v.EndOfLife.Before(t) {
				if v.WaitChannel != nil {
					close(v.WaitChannel)
				}
				delete(c.data, k)
			}
		}
		cacheNewLength := len(c.data)
		c.mutex.Unlock()

		if c.Debug || cacheLength != cacheNewLength {
			log.Printf("Cleaned %s cache in %v, records %d -> %d", c.name, time.Since(t), cacheLength, cacheNewLength)
		}
		time.Sleep(c.cleanUpTime)
	}
}

func (c *Cache) Get(key interface{}) (interface{}, error) {
	var val interface{}
	var waitChannel chan struct{}

	c.mutex.Lock()
	t := time.Now()
	if v, ok := c.data[key]; ok {
		if v.EndOfLife.After(time.Now()) {
			if v.Value != nil {
				val = c.copier(v.Value)
			}
			err := v.Error
			c.mutex.Unlock()
			if c.Debug {
				log.Printf("Got value for %v from %s cache in %v", key, c.name, time.Since(t))
			}
			return val, err
		} else if v.EndOfLife.IsZero() {
			if v.WaitChannel == nil {
				v.WaitChannel = make(chan struct{})
			}
			waitChannel = v.WaitChannel
		} else {
			v.EndOfLife = time.Time{}
		}
	} else {
		c.data[key] = &CacheValue{}
	}
	c.mutex.Unlock()

	if waitChannel != nil {
		if c.Debug {
			log.Printf("Waiting in %s cache queue for %v", c.name, key)
		}
		<-waitChannel
		if c.Debug {
			log.Printf("Waiting in %s cache succeded for %v in %v", c.name, key, time.Since(t))
		}
		c.mutex.Lock()
		if v, ok := c.data[key]; ok {
			if v.Value != nil {
				val = c.copier(v.Value)
			}
			err := v.Error
			c.mutex.Unlock()
			if c.Debug {
				log.Printf("Got value for %v from %s cache in %v", key, c.name, time.Since(t))
			}
			return val, err
		}
		c.mutex.Unlock()
		return nil, fmt.Errorf("cache waited nil result")
	}

	val, err := c.getter(key)
	if c.Debug {
		log.Printf("Got value for %v in %s cache from getter in %v", key, c.name, time.Since(t))
	}

	c.mutex.Lock()
	if v, ok := c.data[key]; ok {
		v.Value = val
		v.Error = err
		if err != nil {
			v.EndOfLife = time.Now().Add(c.BadCacheTime)
		} else {
			v.EndOfLife = time.Now().Add(c.GoodCacheTime)
		}
		if val != nil {
			val = c.copier(val)
		}
		if v.WaitChannel != nil {
			close(v.WaitChannel)
			v.WaitChannel = nil
		}
	} else {
		c.mutex.Unlock()
		return nil, fmt.Errorf("cache key deleted while getting value")
	}
	c.mutex.Unlock()

	return val, err
}

func (c *Cache) Delete(key interface{}) {
	if c.Debug {
		log.Printf("Removing value for %v from %s cache", key, c.name)
	}
	c.mutex.Lock()
	if v, ok := c.data[key]; ok {
		if v.WaitChannel != nil {
			close(v.WaitChannel)
			v.WaitChannel = nil
		}
		delete(c.data, key)
	}
	c.mutex.Unlock()
}

// ==========================================================================================

// Factory of n workers do same job concurrently.
//
// type Factory interface {
//     Do(input []interface{})
//     Stop()
// }
//
type Job struct {
	input     interface{}
	readyChan chan struct{}
	stop      bool
}

type Factory struct {
	WorkersNum uint
	workerFunc Worker
	jobChan    chan *Job
	Debug      bool
}

type Worker func(interface{})

func NewFactory(n uint, w Worker, debug bool) *Factory {
	f := &Factory{
		WorkersNum: n,
		workerFunc: w,
		jobChan:    make(chan *Job, n),
		Debug:      debug,
	}
	for idx := uint(0); idx < n; idx++ {
		go f.worker(idx)
		if f.Debug {
			log.Printf("Worker #%d started", idx)
		}
	}
	return f
}

func (f *Factory) worker(idx uint) {
	var t time.Time
	for {
		j := <-f.jobChan
		if j.stop {
			break
		}
		if f.Debug {
			t = time.Now()
			log.Printf("Worker #%d got the job, %v", idx, j.input)
		}
		f.workerFunc(j.input)
		if f.Debug {
			log.Printf("Worker #%d done the job in %v, %v", idx, time.Since(t), j.input)
		}
		j.readyChan <- struct{}{}
		if f.Debug {
			log.Printf("Worker #%d notified about the job done, %v", idx, j.input)
		}
	}
	if f.Debug {
		log.Printf("Worker #%d stopped", idx)
	}
}

func (f *Factory) Do(input []interface{}) {
	readyChan := make(chan struct{}, len(input))
	for _, j := range input {
		f.jobChan <- &Job{
			input:     j,
			readyChan: readyChan,
		}
	}
	numJobs := len(input)
	for idx := 0; idx < numJobs; idx++ {
		<-readyChan
	}
}

func (f *Factory) Stop() {
	for idx := uint(0); idx < f.WorkersNum; idx++ {
		f.jobChan <- &Job{
			stop: true,
		}
	}
}

// ==========================================================================================

// Send several ticks to one channel with different messages.
//
// type Multicker interface {
//     C chan interface{}
//     Start() error
//     Stop() error
//     Add(string, time.Duration, interface{}) error
//     Delete(string) error
// }
//
const (
	MultickerStopped int32 = iota
	MultickerStarted
)

type SingleTiker struct {
	d    time.Duration
	msg  interface{}
	stop chan struct{}
}

type Multicker struct {
	C     chan interface{}
	Debug bool
	data  map[string]*SingleTiker
	mutex sync.RWMutex
	state int32
}

func NewMulticker(debug bool) *Multicker {
	return &Multicker{
		C:     make(chan interface{}, 1),
		Debug: debug,
		data:  make(map[string]*SingleTiker),
		state: MultickerStopped,
	}
}

func (m *Multicker) Start() error {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	if atomic.CompareAndSwapInt32(&m.state, MultickerStopped, MultickerStarted) {
		if m.Debug {
			log.Printf("Starting multicker")
		}
		for id, t := range m.data {
			if t.stop == nil {
				if m.Debug {
					log.Printf("Starting ticker %s", id)
				}
				t.stop = make(chan struct{}, 1)
				go m.ticker(id, t.d, t.msg, t.stop)
			}
		}
		return nil
	}
	return fmt.Errorf("already started")
}

func (m *Multicker) Stop() error {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	if atomic.CompareAndSwapInt32(&m.state, MultickerStarted, MultickerStopped) {
		if m.Debug {
			log.Printf("Stopping multicker")
		}
		for id, t := range m.data {
			if t.stop != nil {
				if m.Debug {
					log.Printf("Stopping ticker %s", id)
				}
				t.stop <- struct{}{}
				t.stop = nil
			}
		}
	}
	return fmt.Errorf("already stopped")
}

func (m *Multicker) Add(id string, d time.Duration, msg interface{}) error {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	if _, ok := m.data[id]; ok {
		return fmt.Errorf("ticker %s already registered", id)
	}
	if m.Debug {
		log.Printf("Adding ticker %s", id)
	}
	t := &SingleTiker{d: d, msg: msg}
	m.data[id] = t
	if atomic.LoadInt32(&m.state) == MultickerStarted {
		if m.Debug {
			log.Printf("Starting ticker %s", id)
		}
		t.stop = make(chan struct{}, 1)
		go m.ticker(id, d, msg, t.stop)
	}
	return nil
}

func (m *Multicker) Delete(id string) error {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	if v, ok := m.data[id]; ok {
		if v.stop != nil {
			if m.Debug {
				log.Printf("Stopping ticker %s", id)
			}
			v.stop <- struct{}{}
			v.stop = nil
		}
		if m.Debug {
			log.Printf("Deleting ticker %s", id)
		}
		delete(m.data, id)
		return nil
	}
	return fmt.Errorf("no ticker %s registered", id)
}

func (m *Multicker) ticker(id string, d time.Duration, msg interface{}, stop chan struct{}) {
	if m.Debug {
		log.Printf("Entering ticker %s", id)
	}
	ticker := time.NewTicker(d)
	defer ticker.Stop()
	for {
		select {
		case <-ticker.C:
			m.C <- msg
		case <-stop:
			if m.Debug {
				log.Printf("Exiting ticker %s", id)
			}
			return
		}
	}
}

// ==========================================================================================

// Manager acts as a wrapper for an external object, consuming looper interface
//
// type Looper interface {
//     Prepare() error
//     Loop(interface{}) (interface{}, error)
// }
//
// type Manager interface {
//     Start() error
//     Stop() error
//     StopAsync() error // stop from the loop or if no delay needed
//     AddTicker(string, time.Duration, interface{}) error
//     DeleteTicker(string) error
//     Send(interface{}, waitResult bool) (interface{}, error)
// }
//
const (
	ManagerStopped int32 = iota
	ManagerStarting
	ManagerStarted
	ManagerStopping
)

type TickerDesc struct {
	d   time.Duration
	msg interface{}
}

type ControlMsg struct {
	Msg        interface{}
	Error      error
	Time       time.Time
	ResultChan chan *ControlMsg
}

type Looper interface {
	Prepare() error
	Loop(interface{}) (interface{}, error)
}

type Manager struct {
	looper      Looper
	Debug       bool
	state       int32
	multicker   *Multicker
	controlChan chan *ControlMsg
	prepareChan chan error
	stopChan    chan struct{}
}

func NewManager(looper Looper, debug bool) *Manager {
	return &Manager{
		looper:      looper,
		Debug:       debug,
		state:       ManagerStopped,
		multicker:   NewMulticker(debug),
		prepareChan: make(chan error),
		stopChan:    make(chan struct{}),
	}
}

func (m *Manager) Start() error {
	if !atomic.CompareAndSwapInt32(&m.state, ManagerStopped, ManagerStarting) {
		return fmt.Errorf("manager is not stopped")
	}
	if m.Debug {
		log.Printf("Starting manager")
	}
	go m.loop()
	return <-m.prepareChan
}

func (m *Manager) Stop() error {
	if !atomic.CompareAndSwapInt32(&m.state, ManagerStarted, ManagerStopping) {
		return fmt.Errorf("manager is not started")
	}
	if m.Debug {
		log.Printf("Stopping manager")
	}
	m.stopChan <- struct{}{}
	return nil
}

func (m *Manager) StopAsync() error {
	if !atomic.CompareAndSwapInt32(&m.state, ManagerStarted, ManagerStopping) {
		return fmt.Errorf("manager is not started")
	}
	if m.Debug {
		log.Printf("Stopping manager")
	}
	go func() {
		m.stopChan <- struct{}{}
	}()
	return nil
}

func (m *Manager) AddTicker(id string, d time.Duration, msg interface{}) error {
	return m.multicker.Add(id, d, msg)
}

func (m *Manager) DeleteTicker(id string) error {
	return m.multicker.Delete(id)
}

func (m *Manager) Send(msg interface{}, waitResult bool) (resp interface{}, err error) {
	defer func() {
		if r := recover(); r != nil {
			err = fmt.Errorf("manager is stopped")
		}
	}()
	if atomic.LoadInt32(&m.state) != ManagerStarted {
		err = fmt.Errorf("manager is not started")
		return
	}
	var resultChan chan *ControlMsg
	if waitResult {
		resultChan = make(chan *ControlMsg, 1)
	}
	cmOut := &ControlMsg{
		Msg:        msg,
		ResultChan: resultChan,
	}
	if m.Debug {
		cmOut.Time = time.Now()
		log.Printf("Sending message to loop %v", msg)
	}
	m.controlChan <- cmOut
	if waitResult {
		cmIn := <-resultChan
		resp = cmIn.Msg
		err = cmIn.Error
		if m.Debug {
			log.Printf("Recieved message from loop in %v (%v in channel), %v, error %v",
				time.Since(cmIn.Time),
				time.Since(cmOut.Time),
				resp,
				err,
			)
		}
	} else if m.Debug {
		log.Printf("Sent message to loop in %v, %v", time.Since(cmOut.Time), msg)
	}
	return
}

func (m *Manager) loop() {
	defer atomic.StoreInt32(&m.state, ManagerStopped)

	err := m.looper.Prepare()
	m.prepareChan <- err
	if err != nil {
		return
	}
	_ = m.multicker.Start()
	m.controlChan = make(chan *ControlMsg)
	atomic.StoreInt32(&m.state, ManagerStarted)

loop:
	for {
		select {
		case cmIn := <-m.controlChan:
			var msgOut interface{}
			var err error
			var ts time.Time

			if m.Debug {
				ts = time.Now()
				log.Printf("Got message in loop in %v, %v", time.Since(cmIn.Time), cmIn.Msg)
			}
			if atomic.LoadInt32(&m.state) == ManagerStarted {
				msgOut, err = m.looper.Loop(cmIn.Msg)
			} else {
				if m.Debug {
					log.Printf("Skipping message in loop, %v", cmIn.Msg)
				}
				msgOut, err = nil, fmt.Errorf("message skipped in manager: not started")
			}
			if cmIn.ResultChan != nil {
				cmOut := &ControlMsg{
					Msg:   msgOut,
					Error: err,
				}
				if m.Debug {
					cmOut.Time = time.Now()
					log.Printf("Sending message from loop in %v, %v", time.Since(ts), msgOut)
				}
				cmIn.ResultChan <- cmOut
			} else if m.Debug {
				log.Printf("Processed message in loop in %v, %v", time.Since(ts), cmIn.Msg)
			}
		case msgIn := <-m.multicker.C:
			if atomic.LoadInt32(&m.state) != ManagerStarted {
				if m.Debug {
					log.Printf("Skipping tick in loop msg=%v", msgIn)
				}
				break
			}
			var ts time.Time
			if m.Debug {
				ts = time.Now()
				log.Printf("Got tick in loop msg=%v", msgIn)
			}
			msgOut, err := m.looper.Loop(msgIn)
			if m.Debug {
				log.Printf("Processed tick in loop in %v, msgIn=%v, msgOut=%v, err=%v", time.Since(ts), msgIn, msgOut, err)
			}
		case <-m.stopChan:
			break loop
		}
	}
	_ = m.multicker.Stop()

	if m.Debug {
		log.Printf("Closing control channel in loop")
	}
	close(m.controlChan)
}
