package cmd

import (
	"context"
	"fmt"
	"log"
	"os"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/solomon/libs/go/color"
	"a.yandex-team.ru/solomon/tools/metric-downloader/internal/client"
	"a.yandex-team.ru/solomon/tools/metric-downloader/internal/config"
	"a.yandex-team.ru/solomon/tools/metric-downloader/internal/utils"
)

func init() {
	downloadCmd := &cobra.Command{
		Use:   "download <target> [<selector>]",
		Short: "Download metrics from shard target",
		Long: "Usage:\n" +
			"   metric-downloader download <target> [<selector>] [flags]\n" +
			"\nWhere <selector> can be: empty for all metrics or selector, example:\n" +
			"   cluster=\"dev\", service=\"sync\", host==\"solomon-dev-myt-00\"" +
			"\nWhere <target> can be:\n" +
			"   1) project id: {{projectId}}. In that case you will choose needed shard and get metrics from random target in shard\n" +
			"    Example:\n" +
			"          $ metric-downloader download solomon\n" +
			"  \n" +
			"   2) project id and shard id: {{projectId}}/{{shardId}}. You get metrics from random target in shard\n" +
			"    Example:\n" +
			"          $ metric-downloader download solomon/solomon_pre_coremon_spack\n" +
			"  \n" +
			"   3) project id and shard id and target id: {{projectId}}/{{shardId}}/{{targetId}}. You get metrics from specified target in shard\n" +
			"    Example:\n" +
			"          $ metric-downloader download solomon/solomon_pre_coremon_spack/pre-fetcher-man-007.mon.yandex.net\n" +
			"  \n" +
			"   4) fully qualified  id: {{env}}://{{projectId}}/{{shardId}}/{{targetId}}\n" +
			"    Example:\n" +
			"          $ metric-downloader download pre://solomon/solomon_pre_coremon_spack/pre-fetcher-man-007.mon.yandex.net\n" +
			"  \n" +
			"   5) metrics url: {{url}}\n" +
			"    Example:\n" +
			"          $ metric-downloader download http://pre-alerting-sas-00.mon.yandex.net:4530/metrics/alertsStatus\n" +
			"  \n" +
			"   6) with selector\n" +
			"    Example:\n" +
			"          $ metric-downloader download http://sts1-iva-0000.mon.yandex.net:4550/metrics 'host=\"\", type==\"total\"'\n" +
			"  \n" +
			"   7) shard url\n" +
			"    Example:\n" +
			"          $  metric-downloader download https://solomon.cloud.yandex-team.ru/admin/projects/solomon/shards/solomon_cloud_noc_agent\n" +
			"  \n" +
			"Available env: cloud_pre, cloud_prod, test, pre, prod",
		Args: func(cmd *cobra.Command, args []string) error {
			if len(args) != 1 && len(args) != 2 {
				return fmt.Errorf("Usage: " + cmd.Use)
			}
			return nil
		},
		RunE: downloadMetrics,
	}
	rootCmd.AddCommand(downloadCmd)
}

func downloadMetrics(cmd *cobra.Command, args []string) error {
	cfg, err := config.ParseConfig(args)
	if err != nil {
		return err
	}
	log.Printf("Start metric download from: %s\n", color.Blue(formatFromURL(cfg)))

	target := cfg.TargetURL
	if target == "" {
		ctx := context.Background()
		shardClient := client.NewShardClient(cfg.APIURL, cfg.TokenOauth, cfg.TokenIAM)
		target, err = shardClient.GetTargetHost(ctx, cfg)
		if err != nil {
			return err
		}
	}
	log.Printf("Downloading metrics from: %s\n", color.Blue(target))

	devSSHClient, err := client.NewDevSSHClient(target)
	if err != nil {
		return err
	}
	defer devSSHClient.Close()

	metricsInfo, err := devSSHClient.GetMetrics()
	if err != nil {
		return err
	}
	log.Printf("Metrics downloaded to file %s in format %s\n",
		color.Blue(metricsInfo.FilePath), color.Blue(metricsInfo.Format))

	metricInfoConverted, err := utils.ConvertMetrics(metricsInfo)
	if err != nil {
		return err
	}
	defer os.Remove(metricInfoConverted.FilePath)

	err = utils.PrintMetrics(metricInfoConverted, cfg)
	return err
}

func formatFromURL(cfg *config.Config) string {
	if cfg.TargetURL != "" {
		return cfg.TargetURL
	}
	URL := fmt.Sprintf("%s/admin/projects/%s/shards/%s", cfg.APIURL, cfg.ProjectID, cfg.ShardID)
	return URL
}
