package client

import (
	"a.yandex-team.ru/solomon/libs/go/hosts"
	"a.yandex-team.ru/solomon/libs/go/ussh"
	"a.yandex-team.ru/solomon/tools/metric-downloader/internal/utils"
	"fmt"
	"io/ioutil"
	"log"
	"math/rand"
	"net"
	"os"
	"os/exec"
	"time"
)

const (
	devHost = "solomon-dev-myt-00.search.yandex.net"
)

type DevSSHClient struct {
	addresses []hosts.Address
	targetURL string
	tempFile  string
}

func NewDevSSHClient(targetURL string) (*DevSSHClient, error) {
	addresses := make([]hosts.Address, 0, 1)
	ips, err := net.LookupIP(devHost)
	if err != nil {
		return nil, err
	}
	addresses = append(addresses, hosts.Address{Name: devHost, IP: ips[0]})
	return &DevSSHClient{addresses: addresses, targetURL: targetURL}, nil
}

func (c *DevSSHClient) Close() {
	_ = os.Remove(c.tempFile)
}

func (c *DevSSHClient) GetMetrics() (*utils.MetricsInfo, error) {
	folderName := "/var/tmp/metricDownloader" + randomString(8)

	logsDir, err := ioutil.TempDir("", "logs")
	if err != nil {
		return nil, err
	}
	log.Println("SSH client logs in: " + logsDir)
	sshClient := ussh.NewClusterClient(c.addresses, false, logsDir)
	defer sshClient.Close()

	log.Println("Determine metric format by dev host")
	sshClient.RunSequentially("mkdir "+folderName+" "+
		"&& cd "+folderName+" "+
		"&& curl -I -H \"X-Solomon-FetcherId: metric-downloader-client\" '"+c.targetURL+"' > out", 3*time.Minute)

	f, err := ioutil.TempFile("", "temp")
	if err != nil {
		return nil, err
	}
	c.tempFile = f.Name()

	if err := downloadByScp(devHost+":"+folderName+"/out", f.Name()); err != nil {
		return nil, err
	}

	content, err := utils.DetermineFormat(f.Name())
	if err != nil {
		log.Println(err)
		log.Println("Try determine format by GET request")
		sshClient.RunSequentially("cd "+folderName+" "+
			"&& curl -i -H \"X-Solomon-FetcherId: metric-downloader-client\" '"+c.targetURL+"' > out", 3*time.Minute)

		if err := downloadByScp(devHost+":"+folderName+"/out", f.Name()); err != nil {
			return nil, err
		}

		content, err = utils.DetermineFormat(f.Name())
		if err != nil {
			return nil, fmt.Errorf("can't determine response format")
		}
	}

	log.Println("Download metrics by dev host")
	sshClient.RunSequentially("cd "+folderName+" "+
		"&& curl -H \"X-Solomon-FetcherId: metric-downloader-client\" '"+c.targetURL+"' > out", 3*time.Minute)

	if err := downloadByScp(devHost+":"+folderName+"/out", f.Name()); err != nil {
		return nil, err
	}

	sshClient.RunSequentially("rm -fr "+folderName, 3*time.Minute)

	return &utils.MetricsInfo{
		Format:   content,
		FilePath: f.Name(),
	}, nil
}

func downloadByScp(from string, to string) error {
	cmd := exec.Command("scp", "-r", from, to)
	return cmd.Run()
}

const letterBytes = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"

func randomString(n int) string {
	rand.Seed(time.Now().UTC().UnixNano())
	b := make([]byte, n)
	for i := range b {
		b[i] = letterBytes[rand.Intn(len(letterBytes))]
	}
	return string(b)
}
