package client

import (
	"bytes"
	"context"
	"fmt"
	"log"
	"strconv"
	"time"

	"a.yandex-team.ru/solomon/libs/go/color"
	"a.yandex-team.ru/solomon/libs/go/uhttp"
	"a.yandex-team.ru/solomon/tools/metric-downloader/internal/config"
)

const (
	baseProjectsURL = "/api/v3/projects/"
	shardsPath      = "/shards/"
	targetsPath     = "/targets/"
)

type ShardClient struct {
	client *uhttp.Client
}

type target struct {
	URL  string `json:"url"`
	Host string `json:"host"`
	Dc   string `json:"dc"`
}
type respTargets struct {
	Targets []target `json:"targets"`
}

type Shard struct {
	ID string `json:"id"`
}

type respShards struct {
	Shards []Shard `json:"shards"`
}

func NewShardClient(baseURL, tokenOauth string, tokenIam string) *ShardClient {
	var token uhttp.Token
	if tokenIam == "" {
		token = uhttp.OAuthToken(tokenOauth)
	} else {
		token = uhttp.IamToken(tokenIam)
	}
	return &ShardClient{client: uhttp.NewClient(baseURL+baseProjectsURL, "", token, 20*time.Second, false)}
}

func (c *ShardClient) GetTargetHost(ctx context.Context, config *config.Config) (string, error) {
	shardID, err := c.askShardID(ctx, config)
	if err != nil {
		return "", err
	}

	targetID, err := c.getFirstTarget(ctx, shardID, config)
	if err != nil {
		return "", err
	}

	return targetID, nil
}

func (c *ShardClient) getFirstTarget(ctx context.Context, shardID string, config *config.Config) (string, error) {
	if config.TargetHOST != "" {
		url, err := c.getTargetURL(ctx, config.TargetHOST, shardID, config.ProjectID)
		if err != nil {
			return "", err
		}
		return url, nil
	}
	req, err := c.client.NewGetRequest(ctx, config.ProjectID+shardsPath+shardID+targetsPath+"?pageSize=1")
	if err != nil {
		return "", fmt.Errorf("cannot create request: %v", err)
	}

	var resp respTargets
	err = c.client.SendJSONRequest(req, &resp)
	if err != nil {
		return "", err
	}
	if len(resp.Targets) == 0 {
		return "", fmt.Errorf("no targets in project %s and shard %s", config.ProjectID, shardID)
	}
	log.Printf("Target host %s (%s) was selected by random\n",
		color.Blue(resp.Targets[0].Host), color.Blue(resp.Targets[0].Dc))
	return resp.Targets[0].URL, nil
}

func (c *ShardClient) getTargetURL(ctx context.Context, targetID string, shardID string, projectID string) (string, error) {
	req, err := c.client.NewGetRequest(ctx, projectID+shardsPath+shardID+targetsPath)
	if err != nil {
		return "", fmt.Errorf("cannot create request: %v", err)
	}

	var resp respTargets
	err = c.client.SendJSONRequest(req, &resp)
	if err != nil {
		return "", err
	}
	for i := 0; i < len(resp.Targets); i++ {
		if resp.Targets[i].Host == targetID {
			return resp.Targets[i].URL, nil
		}
	}
	return "", fmt.Errorf("target %s not found", targetID)
}

func (c *ShardClient) askShardID(ctx context.Context, config *config.Config) (string, error) {
	if config.ShardID != "" {
		return config.ShardID, nil
	}
	req, err := c.client.NewGetRequest(ctx, config.ProjectID+shardsPath+"?pageSize=1000")
	if err != nil {
		return "", fmt.Errorf("cannot create request: %v", err)
	}

	var resp respShards
	err = c.client.SendJSONRequest(req, &resp)
	if err != nil {
		return "", err
	}
	if len(resp.Shards) == 0 {
		return "", fmt.Errorf("no shards in project %s", config.ProjectID)
	}
	if len(resp.Shards) == 1 {
		return resp.Shards[0].ID, nil
	}
	var buffer bytes.Buffer
	buffer.WriteString("\nShards list:\n")
	for i := 0; i < len(resp.Shards); i++ {
		buffer.WriteString("\n" + strconv.Itoa(i) + ". " + resp.Shards[i].ID)
	}
	buffer.WriteString("\n\nShard not specified, enter number of target shard:\n")
	log.Println(buffer.String())
	index, err := enterInt(len(resp.Shards))
	if err != nil {
		return "", err
	}
	log.Printf("Shard %s was selected\n", color.Blue(resp.Shards[index].ID))
	return resp.Shards[index].ID, nil
}

func enterInt(max int) (int64, error) {
	for {
		var v string
		_, err := fmt.Scanln(&v)
		if err != nil {
			return 0, err
		}
		res, err := strconv.ParseInt(v, 10, 64)
		if err != nil {
			log.Println(err)
			continue
		}
		if res < 0 || res >= int64(max) {
			log.Printf("value must be between 0 and %d\n", max)
			continue
		}
		return res, nil
	}
}
