package config

import (
	"a.yandex-team.ru/library/go/yandex/oauth"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
	"a.yandex-team.ru/solomon/libs/go/iam"
	"a.yandex-team.ru/solomon/libs/go/selector"
	"a.yandex-team.ru/solomon/protos/secrets"
	"context"
	"fmt"
	"log"
	"os"
	"os/user"
	"strings"
	"time"
)

const (
	cloudPreEnv        = "cloud_pre"
	cloudPreprodAPIURL = "https://solomon.cloud-preprod.yandex-team.ru"
	cloudProdEnv       = "cloud_prod"
	cloudProdAPIURL    = "https://solomon.cloud.yandex-team.ru"
	testEnv            = "test"
	testAPIURL         = "https://solomon-test.yandex-team.ru"
	preEnv             = "pre"
	preprodAPIURL      = "https://solomon-pre.yandex-team.ru"
	prodEnv            = "prod"
	prodAPIURL         = "https://solomon.yandex-team.ru"
	clientID           = "e397b846943844228aca41877a8b85d6"
	clientSecret       = "aa15736d8bc8492c8bfa0578d9892f3f"
)

type Config struct {
	TokenIAM        string
	TokenOauth      string
	APIURL          string
	ShardID         string
	ProjectID       string
	TargetHOST      string
	TargetURL       string
	Selectors       *selector.Selectors
	SelectorsString string
}

func ParseConfig(args []string) (*Config, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 20*time.Second)
	defer cancel()

	config := &Config{
		APIURL: prodAPIURL,
	}
	tokenOauth, err := getOauthToken(ctx)
	if err != nil {
		return nil, err
	}
	config.TokenOauth = tokenOauth

	env := int32(secrets.CloudEnv_PROD)
	from := args[0]

	err = parseParam(from, config, &env)
	if err != nil {
		return nil, err
	}
	t, err := getIamToken(ctx, env, "IAM_SECRET_FROM")
	if err != nil && config.TokenOauth == "" {
		return nil, err
	} else if err != nil {
		log.Println("OAuth would be used in client")
	}
	config.TokenIAM = t
	if len(args) == 2 {
		config.Selectors, err = selector.ParseSelectors(args[1])
		if err != nil {
			return nil, err
		}
		config.SelectorsString = args[1]
	} else {
		config.Selectors, _ = selector.ParseSelectors("")
		config.SelectorsString = ""
	}

	return config, nil
}

func getIamToken(ctx context.Context, env int32, secretVariable string) (string, error) {
	token, err := iam.GetIamTokenFromLocalMetaDataService()
	if err != nil {
		return "", fmt.Errorf("unable to get IAM token from local metadata service: %w", err)
	}
	if token != "" {
		return token, nil
	}
	keyJSON, err := loadYavKey(ctx, os.Getenv(secretVariable))
	if err != nil {
		return "", err
	}
	token, err = iam.GetIamToken(secrets.CloudEnv(env), keyJSON)
	if err != nil {
		return "", err
	}
	return token, nil
}

func getOauthToken(ctx context.Context) (string, error) {
	tokenOauth := os.Getenv("OAUTH_TOKEN")
	var err error
	if tokenOauth == "" {
		tokenOauth, err = oauth.GetTokenBySSH(ctx, clientID, clientSecret)
		if err != nil {
			return "", err
		}
		log.Println("OAuth token received with API")
	}
	return tokenOauth, nil
}

func parseParam(val string, config *Config, env *int32) error {
	if strings.HasPrefix(val, "http") {
		if isSolomonShardURL(val) {
			vals := strings.Split(val, "/")
			if len(vals) >= 8 {
				config.ProjectID = vals[5]
				config.ShardID = vals[7]
				config.APIURL = strings.Split(val, "/admin")[0]
				return nil
			}
		}
		config.TargetURL = val
		return nil
	}
	if strings.Contains(val, "://") {
		vals := strings.Split(val, "://")
		parseTarget(vals[1], config)
		if vals[0] == cloudPreEnv {
			config.APIURL = cloudPreprodAPIURL
			*env = int32(secrets.CloudEnv_PREPROD)
		} else if vals[0] == cloudProdEnv {
			config.APIURL = cloudProdAPIURL
			*env = int32(secrets.CloudEnv_PROD)
		} else if vals[0] == preEnv {
			config.APIURL = preprodAPIURL
			*env = 0
		} else if vals[0] == prodEnv {
			config.APIURL = prodAPIURL
			*env = 0
		} else if vals[0] == testEnv {
			config.APIURL = testAPIURL
			*env = 0
		} else {
			return fmt.Errorf("unknown env %s", vals[0])
		}
	} else {
		parseTarget(val, config)
	}
	return nil
}

func isSolomonShardURL(val string) bool {
	return strings.HasPrefix(val, prodAPIURL) || strings.HasPrefix(val, preprodAPIURL) ||
		strings.HasPrefix(val, testAPIURL) || strings.HasPrefix(val, cloudProdAPIURL) || strings.HasPrefix(val, cloudPreprodAPIURL)
}

func parseTarget(val string, config *Config) {
	vals := strings.Split(val, "/")
	config.ProjectID = vals[0]
	if len(vals) > 1 {
		config.ShardID = vals[1]
	}
	if len(vals) > 2 {
		config.TargetHOST = vals[2]
	}
}

// below code is copied from "a.yandex-team.ru/solomon/tools/secrets"
func currentLogin() (string, error) {
	sysUser, err := user.Current()
	if err != nil {
		return "", err
	}
	return sysUser.Username, nil
}

func loadYavKey(ctx context.Context, secretKey string) (string, error) {
	secretKeyArr := strings.Split(secretKey, "/")
	if len(secretKeyArr) != 2 {
		return "", fmt.Errorf("invalid format of secret key: %s", secretKey)
	}

	login, err := currentLogin()
	if err != nil {
		return "", fmt.Errorf("cannot get login of current user, %v", err)
	}

	token, err := oauth.GetTokenBySSH(ctx, clientID, clientSecret, oauth.WithUserLogin(login))
	if err != nil {
		return "", fmt.Errorf("cannot get OAuth token by SSH, %v", err)
	}

	client, err := httpyav.NewClient(httpyav.WithOAuthToken(token))
	if err != nil {
		return "", fmt.Errorf("cannot initialize yav client, %v", err)
	}

	ver, err := client.GetVersion(ctx, secretKeyArr[0])
	if err != nil {
		return "", fmt.Errorf("cannot load sercret %s, %v", secretKeyArr[0], err)
	}

	for _, value := range ver.Version.Values {
		if value.Key == secretKeyArr[1] {
			return value.Value, nil
		}
	}

	return "", fmt.Errorf("cannot find %s in secret %s", secretKeyArr[1], secretKeyArr[0])
}
