package utils

import (
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"strings"

	"a.yandex-team.ru/solomon/libs/go/color"
)

const (
	json       = "application/json"
	txt        = "application/x-solomon-txt"
	spack      = "application/x-solomon-spack"
	prometheus = "text/plain"

	toText   = "to-text"
	fromProm = "from-prom"
	fromJSON = "from-json"
)

type MetricsInfo struct {
	Format   string
	FilePath string
}

func ConvertMetrics(metricInfo *MetricsInfo) (*MetricsInfo, error) {
	if metricInfo.Format == txt {
		return metricInfo, nil
	}
	log.Printf("Metrics will be converted to %s format\n", color.Blue(txt))
	if metricInfo.Format == json {
		return multiConvert(metricInfo, fromJSON)
	} else if metricInfo.Format == prometheus {
		return multiConvert(metricInfo, fromProm)
	} else if metricInfo.Format == spack {
		return multiConvert(metricInfo, "")
	}
	return nil, fmt.Errorf("unknown converting format: %s", color.Blue(metricInfo.Format))
}

func multiConvert(metricInfo *MetricsInfo, from string) (*MetricsInfo, error) {
	fSpack, err := ioutil.TempFile("", "temp")
	if err != nil {
		return nil, err
	}
	defer os.Remove(fSpack.Name())
	path, err := convert(metricInfo.FilePath, from, fSpack.Name())
	if err != nil {
		return nil, err
	}
	fTxt, err := ioutil.TempFile("", "temp")
	if err != nil {
		return nil, err
	}
	path, err = convert(path, toText, fTxt.Name())
	if err != nil {
		return nil, err
	}
	log.Printf("Metrics converted, output file %s\n", color.Blue(path))
	return &MetricsInfo{
		FilePath: path,
		Format:   txt,
	}, nil
}

func convert(path string, param string, outPath string) (string, error) {
	if param == "" {
		return path, nil
	}
	spack := exec.Command("spack", param, path)
	outfile, err := os.Create(outPath)
	if err != nil {
		return "", err
	}
	defer outfile.Close()
	spack.Stdout = outfile
	err = spack.Run()
	if err != nil {
		return "", fmt.Errorf("spack conversion error: %s\nbuild spack tool and add to PATH variable: ya make ../spack", err)
	}
	return outPath, nil
}

func DetermineFormat(file string) (string, error) {
	bytes, err := ioutil.ReadFile(file)
	if err != nil {
		return "", err
	}
	headers := string(bytes)
	if strings.Contains(headers, json) {
		return json, nil
	} else if strings.Contains(headers, txt) {
		return txt, nil
	} else if strings.Contains(headers, spack) {
		return spack, nil
	} else if strings.Contains(headers, prometheus) {
		return prometheus, nil
	}
	return "", fmt.Errorf("unsupported format in headers:\n %s", headers)
}
