package utils

import (
	"bufio"
	"fmt"
	"log"
	"os"
	"strings"

	"a.yandex-team.ru/solomon/libs/go/color"
	"a.yandex-team.ru/solomon/tools/metric-downloader/internal/config"
)

var logger = log.New(os.Stdout, "", 0)

func PrintMetrics(metricInfo *MetricsInfo, cfg *config.Config) error {
	log.Printf("Print metrics for selector %s:\n", color.Blue(cfg.SelectorsString))
	file, err := os.Open(metricInfo.FilePath)
	if err != nil {
		return err
	}
	defer file.Close()

	commonLabels := make(map[string]string)
	scanner := bufio.NewScanner(file)
	anyPrinted := false
	for scanner.Scan() {
		line := scanner.Text()
		if strings.HasPrefix(line, "common labels: {") {
			addCommonLabels(line, commonLabels)
		} else if strings.HasPrefix(line, "common ") {
			continue
		} else {
			lbls := make(map[string]string)
			for k, v := range commonLabels {
				lbls[k] = v
			}
			addLabels(line, lbls)
			match := cfg.Selectors.Match(lbls)
			if match {
				logger.Println(line)
				anyPrinted = true
			}
		}
	}
	if !anyPrinted {
		log.Println(color.Red(fmt.Sprintf("No metrics for selector: %s", cfg.SelectorsString)))
	}
	return scanner.Err()
}

func addCommonLabels(line string, lbls map[string]string) {
	prepare := strings.TrimPrefix(line, "common labels: {")
	prepare = strings.TrimSuffix(prepare, "}")
	addParts(prepare, lbls)
}

func addParts(lblsStr string, lbls map[string]string) {
	parts := strings.Split(lblsStr, ", ")
	for _, part := range parts {
		if part == "" {
			continue
		}
		vals := strings.SplitN(part, "=", 2)
		lbls[vals[0]] = prepareValue(vals[1])
	}
}

/* starts with:
 HIST_RATE auth.elapsedTimeMs{authType='TvmService', host='1'} [{16: 0, 32: 0, inf: 0}]
after trim:
HIST_RATE auth.elapsedTimeMs{authType='TvmService', host='1'} [{16: 0, 32: 0, inf: 0}]
after split by ' ':
[HIST_RATE, auth.elapsedTimeMs{authType='TvmService', host='1'} [{16: 0, 32: 0, inf: 0}]]
after split by '{':
[auth.elapsedTimeMs, authType='TvmService', host='1'} [{16: 0, 32: 0, inf: 0}]]
after substring with '} [':
[authType='TvmService', host='1']
*/
func addLabels(line string, lbls map[string]string) {
	lineTrimmedSpaces := strings.Trim(line, " ")
	metricSplitValues := strings.SplitN(lineTrimmedSpaces, " ", 2)
	metric := strings.SplitN(metricSplitValues[1], "{", 2)
	if len(metric) < 2 {
		return
	}
	lbls["sensor"] = metric[0]
	idx := strings.LastIndex(metric[1], "} [")
	addParts(metric[1][:idx], lbls)
}

func prepareValue(value string) string {
	return strings.Replace(value, "'", "", -1)
}
