package ru.yandex.solomon.tool;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import com.google.common.collect.Lists;
import com.yandex.ydb.auth.tvm.TvmAuthContext;
import com.yandex.ydb.auth.tvm.YdbClientId;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.scheme.SchemeOperationProtos.Entry;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableColumn;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.transaction.TxControl;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.StructType;
import com.yandex.ydb.table.values.Type;
import com.yandex.ydb.table.values.Value;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.tool.cfg.SolomonCluster;

/**
 * Recursively copies all directories and tables from one YDB instance to another.
 *
 * @author Sergey Polovko
 */
public class CopyConfigs {

    private static final class TablePath {
        final String rootPath;
        final String dirPath;
        final String name;

        TablePath(String rootPath, String dirPath, String name) {
            this.rootPath = rootPath;
            this.dirPath = dirPath;
            this.name = name;
        }

        TablePath newPath(String newRoot) {
            return new TablePath(newRoot, dirPath, name);
        }

        @Override
        public String toString() {
            return rootPath + '/' + dirPath + '/' + name;
        }
    }

    private static final class TableData {
        final StructType rowType;
        final List<Value> rows = new ArrayList<>();

        TableData(TableDescription tableDesc) {
            List<TableColumn> columns = tableDesc.getColumns();
            Map<String, Type> members = new HashMap<>(columns.size());
            for (TableColumn column : columns) {
                members.put(column.getName(), column.getType());
            }
            this.rowType = StructType.of(members);
        }

        void addRow(ResultSetReader resultSet) {
            var fields = new HashMap<String, Value>(resultSet.getColumnCount());
            for (int i = 0; i < resultSet.getColumnCount(); i++) {
                fields.put(resultSet.getColumnName(i), resultSet.getColumn(i).getValue());
            }
            rows.add(rowType.newValue(fields));
        }
    }

    private static final class Source implements AutoCloseable {
        final String rootPath;
        final TableClient tableClient;
        final SchemeClient schemeClient;
        final SessionRetryContext retryCtx;

        Source(SolomonCluster cluster) {
            this.rootPath = cluster.kikimrRootPath();
            var transport = YdbHelper.createRpcTransport(cluster);
            this.tableClient = YdbHelper.createTableClient(transport);
            this.schemeClient = YdbHelper.createSchemaClient(transport);

            this.retryCtx = SessionRetryContext.create(tableClient)
                .maxRetries(10)
                .build();
        }

        @Override
        public void close() {
            this.schemeClient.close();
            this.tableClient.close();
        }

        List<TablePath> findAllTables() {
            List<TablePath> tables = new ArrayList<>();

            Queue<String> queue = new LinkedList<>();
            queue.offer(rootPath);

            while (!queue.isEmpty()) {
                String dir = queue.poll();

                var entries = schemeClient.listDirectory(dir)
                    .join()
                    .expect("cannot list directory " + dir)
                    .getChildren();

                for (Entry e : entries) {
                    if (e.getType() == Entry.Type.TABLE) {
                        tables.add(new TablePath(rootPath, dir.substring(rootPath.length() + 1), e.getName()));
                    } else if (e.getType() == Entry.Type.DIRECTORY) {
                        queue.offer(dir + '/' + e.getName());
                    } else {
                        System.err.println("skip " + dir + '/' + e.getName() + ", which has type " + e.getType());
                    }
                }
            }

            return tables;
        }

        TableDescription describeTable(TablePath path) {
            return retryCtx.supplyResult(session -> session.describeTable(path.toString()))
                .join()
                .expect("cannot describe table " + path);
        }

        TableData readAll(TablePath path, TableDescription tableDesc) {
            var settings = ReadTableSettings.newBuilder()
                .timeout(2, TimeUnit.MINUTES)
                .orderedRead(true)
                .build();

            return retryCtx.supplyResult(session -> {
                    TableData tableData = new TableData(tableDesc);
                    return session.readTable(path.toString(), settings, resultSet -> {
                            while (resultSet.next()) {
                                tableData.addRow(resultSet);
                            }
                        })
                        .thenApply(s -> s.isSuccess() ? Result.success(tableData) : Result.fail(s));
                })
                .join()
                .expect("cannot read table " + path);
        }
    }

    private static final class Target implements AutoCloseable {
        final String database;
        final TvmAuthContext authContext;
        final TableClient tableClient;
        final SchemeClient schemeClient;
        final SessionRetryContext retryCtx;
        final Set<String> createdPaths = new HashSet<>();

        Target(String endpoint, String database, int clientId, String clientSecret) {
            this.database = database;
            this.authContext = TvmAuthContext.useTvmApi(clientId, clientSecret);

            var transport = GrpcTransport.forEndpoint(endpoint, database)
                .withAuthProvider(authContext.authProvider(YdbClientId.YDB))
                .build();

            this.tableClient = YdbHelper.createTableClient(transport);
            this.schemeClient = YdbHelper.createSchemaClient(transport);

            this.retryCtx = SessionRetryContext.create(tableClient)
                .maxRetries(10)
                .build();
        }

        @Override
        public void close() {
            this.schemeClient.close();
            this.tableClient.close();
            this.authContext.close();
        }

        void createTable(TablePath path, TableDescription tableDesc) {
            if (!createdPaths.contains(path.dirPath)) {
                String prefix = path.rootPath;
                for (String dir : StringUtils.split(path.dirPath, '/')) {
                    String dirPath = prefix + '/' + dir;
                    System.out.println("create dir " + dirPath);
                    schemeClient.makeDirectory(dirPath)
                        .join()
                        .expect("cannot create directory " + dirPath);
                    prefix = dirPath;
                }

                createdPaths.add(path.dirPath);
            }

            retryCtx.supplyStatus(session -> session.createTable(path.toString(), tableDesc))
                .join()
                .expect("cannot create table " + path);
        }

        void writeAll(TablePath path, TableData data) {
            ListType rowsType = ListType.of(data.rowType);
            String query = String.format("""
                    --!syntax_v1
                    DECLARE $rows AS %s;
                    REPLACE INTO `%s` SELECT * FROM AS_TABLE($rows);
                    """, rowsType, path);

            for (List<Value> partition : Lists.partition(data.rows, 300)) {
                Params params = Params.of("$rows", rowsType.newValue(partition));
                retryCtx.supplyResult(session -> session.executeDataQuery(query, TxControl.serializableRw(), params))
                    .join()
                    .expect("cannot insert data into " + path);
            }
        }

        void removeAll() {
            removeAll(database, 0);
        }

        void removeAll(String dir, int level) {
            var entries = schemeClient.listDirectory(dir)
                .join()
                .expect("cannot list directory " + dir)
                .getChildren();

            for (Entry e : entries) {
                if (e.getName().startsWith(".")) {
                    continue;
                }

                String path = dir + '/' + e.getName();
                if (e.getType() == Entry.Type.DIRECTORY) {
                    removeAll(path, level + 1);
                } else if (e.getType() == Entry.Type.TABLE) {
                    System.out.println("drop table " + path);
                    retryCtx.supplyStatus(s -> s.dropTable(path))
                        .join()
                        .expect("cannot drop table " + path);
                }
            }

            if (level != 0) {
                // drop all subdirectories except root dir
                System.out.println("drop directory " + dir);
                schemeClient.removeDirectory(dir)
                    .join()
                    .expect("cannot remove directory " + dir);
            }
        }
    }

    public static void main(String[] args) {
        if (args.length != 2) {
            System.err.println("Usage: tool {pre|test} <tvm_secret>");
            System.err.println();
            System.err.println("Prestable secret https://yav.yandex-team.ru/secret/sec-01dq7m9a39vnxwj1xht0264ctd/explore/version/ver-01dq7m9a9de7w1d9nfbtptt0jq");
            System.err.println("Testing secret https://yav.yandex-team.ru/secret/sec-01dq7m99vxbhb20ddbfdgszdhv/explore/version/ver-01dq7m9a24znq5cvbzcch9anns");
            System.exit(1);
        }

        final Source source;
        final Target target;

        if (args[0].equals("pre")) {
            source = new Source(SolomonCluster.PRESTABLE_FRONT);
            target = new Target("ydb-ru-prestable.yandex.net:2135", "/ru-prestable/solomon/prestable/solomon", 2010240, args[1]);
        } else if (args[0].equals("test")) {
            source = new Source(SolomonCluster.TEST_FRONT);
            target = new Target("ydb-ru-prestable.yandex.net:2135", "/ru-prestable/solomon/development/solomon", 2010238, args[1]);
        } else {
            System.err.println("unknown environment: " + args[0]);
            System.exit(1);
            return;
        }

        target.removeAll();

        try (source; target) {
            List<TablePath> sourceTables = source.findAllTables();
            for (TablePath sourcePath : sourceTables) {
                TablePath targetPath = sourcePath.newPath(target.database);

                System.out.println("creating table " + targetPath);
                TableDescription tableDesc = source.describeTable(sourcePath);
                target.createTable(targetPath, tableDesc);

                System.out.println("copying data " + sourcePath + " -> " + targetPath);
                target.writeAll(targetPath, source.readAll(sourcePath, tableDesc));
            }

            System.exit(0);
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }
    }
}
