package ru.yandex.solomon.tool;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.metabase.client.MetabaseClient;
import ru.yandex.metabase.client.MetabaseClientOptions;
import ru.yandex.metabase.client.MetabaseClients;
import ru.yandex.misc.cmdline.CmdArgsChief;
import ru.yandex.misc.cmdline.CmdLineArgumentsException;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyResponse;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.FindRequest;
import ru.yandex.solomon.metabase.api.protobuf.FindResponse;
import ru.yandex.solomon.metabase.api.protobuf.Metric;
import ru.yandex.solomon.metabase.api.protobuf.TSliceOptions;
import ru.yandex.solomon.util.PropertyInitializer;

import static ru.yandex.solomon.tool.cfg.SolomonPorts.COREMON_GRPC;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class DeleteMetricsMain {
    private static final int MAX_DELETE_ATTEMPT = 10;
    private static final int MAX_BATCH_SIZE = 10_000;

    static {
        PropertyInitializer.init();
    }

    private static final Map<String, List<String>> addressesByEnv = Map.of(
        "PROD", List.of(
                "conductor_group://solomon_prod_fetcher_sas:" + COREMON_GRPC,
                "conductor_group://solomon_prod_fetcher_vla:" + COREMON_GRPC),

        "PRE", List.of("conductor_group://solomon_pre_fetcher:" + COREMON_GRPC),

        "TEST", List.of("conductor_group://solomon_test_meta_storage:" + COREMON_GRPC),

        "CLOUD_PRE", List.of("conductor_group://cloud_preprod_solomon-core:" + COREMON_GRPC),

        "CLOUD_PROD", List.of(
                "conductor_group://cloud_prod_solomon-core_sas:" + COREMON_GRPC,
                "conductor_group://cloud_prod_solomon-core_vla:" + COREMON_GRPC)
    );

    public static void main(String[] stringArgs) {
        DeleteMetricsArgs args = new DeleteMetricsArgs();
        try {
            new CmdArgsChief(args, stringArgs);
        } catch (CmdLineArgumentsException e) {
            System.err.println(e.getMessage());
            System.exit(0);
        }

        System.exit(main(args));
    }

    private static int main(DeleteMetricsArgs args) {
        try {
            var globalSelectors = Selectors.parse(args.selectors);
            var shardSelector = ShardSelectors.onlyShardKey(globalSelectors);
            var metricSelector = ShardSelectors.withoutShardKey(globalSelectors);

            if (shardSelector.isEmpty()) {
                System.err.printf("Specified not valid selector %s not specified shard selector\n", globalSelectors);
                return 1;
            }

            List<String> addresses = addressesByEnv.get(args.env);
            if (addresses == null) {
                System.err.printf("Unspecified environment: %s\n", args.env);
                return 1;
            }


            List<MetabaseClient> metabaseClients = createMetabaseClients(addresses);
            var shards = matchShards(metabaseClients, shardSelector);
            if (!args.delete) {
                for (var shard : shards) {
                    var localSelector =  ShardSelectors.concat(shard, metricSelector);
                    for (MetabaseClient metabaseClient : metabaseClients) {
                        metabaseClient.forceUpdateClusterMetaData().join();
                        int offset = 0;
                        do {
                            List<Metric> resolvedMetrics = resolveMetrics(metabaseClient, localSelector, offset, true);
                            if (resolvedMetrics.isEmpty()) {
                                break;
                            }
                            offset += resolvedMetrics.size();
                        } while (args.continueDeletion);
                    }
                }

                return 0;
            }

            for (var shard : shards) {
                var localSelector =  ShardSelectors.concat(shard, metricSelector);
                for (MetabaseClient metabaseClient : metabaseClients) {
                    metabaseClient.forceUpdateClusterMetaData().join();
                    do {
                        List<Metric> resolvedMetrics = resolveMetrics(metabaseClient, localSelector, args.verbose);
                        if (resolvedMetrics.isEmpty()) {
                            break;
                        }
                        deleteMetrics(metabaseClient, resolvedMetrics, args.verbose);
                    } while (args.continueDeletion);
                }
            }

            return 0;
        } catch (Throwable t) {
            t.printStackTrace();
            return 1;
        }
    }

    private static Set<ShardKey> matchShards(List<MetabaseClient> clients, Selectors shardSelector) {
        // fast path
        {
            ShardKey key = ShardSelectors.getShardKeyOrNull(shardSelector);
            if (key != null) {
                return Set.of(key);
            }
        }

        Set<ShardKey> shards = new HashSet<>();
        for (var client : clients) {
            client.forceUpdateClusterMetaData().join();
            client.shards(shardSelector).forEach(labels -> shards.add(ShardKey.get(labels)));
        }
        return shards;
    }

    private static List<Metric> resolveMetrics(MetabaseClient metabaseClient, Selectors selectors, boolean b) {
        return resolveMetrics(metabaseClient, selectors, 0, b);
    }

    private static List<Metric> resolveMetrics(MetabaseClient metabaseClient, Selectors selectors, int offset, boolean verbose) {
        List<Metric> metrics = syncSearchMetrics(metabaseClient, selectors, offset);
        System.err.printf("# Resolved %d metrics by selectors %s\n", metrics.size(), selectors);
        if (verbose) {
            printMetrics(metrics);
        }
        return metrics;
    }

    private static void deleteMetrics(MetabaseClient metabaseClient, Collection<Metric> metrics, boolean verbose) {
        List<Metric> removedMetrics = syncDeleteMetrics(metabaseClient, metrics);
        System.err.printf("# Deleted %d metrics from Metabase\n", removedMetrics.size());
        if (verbose) {
            printMetrics(removedMetrics);
        }
    }

    private static void printMetrics(List<Metric> metrics) {
        for (Metric metric : metrics) {
            System.err.printf("%s/%s - %s\n",
                metric.getMetricId().getShardId(),
                metric.getMetricId().getLocalId(),
                LabelConverter.protoToLabels(metric.getLabelsList()));
        }
    }

    /**
     * Delete metrics from metabase and stockpile by one call, metabase as a first action will
     * delete metrics from stockpile after that from metabase
     */
    private static List<Metric> syncDeleteMetrics(MetabaseClient metabaseClient, Collection<Metric> metrics) {
        int attempt = 0;
        while (true) {
            DeleteManyResponse response = metabaseClient.deleteMany(DeleteManyRequest.newBuilder()
                    .addAllMetrics(metrics)
                    .setDeadlineMillis(System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(3))
                    .build())
                    .join();

            if (response.getStatus() == EMetabaseStatusCode.OK) {
                return response.getMetricsList();
            }

            if (attempt++ == MAX_DELETE_ATTEMPT) {
                throw new IllegalStateException(response.getStatus() + ": " + response.getStatusMessage());
            }

            System.err.println("Delete failed " + response.getStatus() + ": " + response.getStatusMessage());
        }
    }

    private static List<Metric> syncSearchMetrics(MetabaseClient metabaseClient, Selectors selectors, int offset) {
        FindResponse response = metabaseClient.find(FindRequest.newBuilder()
                .addAllSelectors(LabelSelectorConverter.selectorsToProto(selectors))
                .setDeadlineMillis(System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(3))
                .setSliceOptions(TSliceOptions.newBuilder()
                    .setOffset(offset)
                    .setLimit(MAX_BATCH_SIZE)
                    .build())
                .build())
                .join();

        if (response.getStatus() != EMetabaseStatusCode.OK) {
            throw new IllegalStateException(response.getStatus() + ": " + response.getStatusMessage());
        }

        return response.getMetricsList();
    }

    private static List<MetabaseClient> createMetabaseClients(List<String> addresses) {
        var options = MetabaseClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                        .setRequestTimeOut(3, TimeUnit.MINUTES)
                        .setKeepAliveDelay(1, TimeUnit.MINUTES)
                        .setKeepAliveTimeout(1, TimeUnit.SECONDS))
                .setExpireClusterMetadata(30, TimeUnit.SECONDS)
                .build();

        return addresses.stream()
            .map(hosts -> MetabaseClients.createDynamic(List.of(hosts), options))
            .collect(Collectors.toList());
    }
}
