package ru.yandex.solomon.tool;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.stream.IntStream;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.core.db.dao.ClustersDao;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbClustersDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbShardsDao;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.tool.cfg.SolomonCluster;

/**
 * @author Sergey Polovko
 */
public class DropCloudFolders implements AutoCloseable {

    public static final int THREADS = 16;

    public static void main(String[] args) {
        if (args.length != 2) {
            System.out.println("Usage: tool {preprod|prod} <file>");
            System.exit(1);
        }

        try {
            SolomonCluster cluster = cluster(args[0]);
            Path file = Path.of(args[1]);

            try (var tool = new DropCloudFolders(cluster)) {
                tool.run(file);
            }
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }

        System.exit(0);
    }

    private YdbClient ydbClient;
    private String root;

    private final ClustersDao clustersDao;
    private final ShardsDao shardsDao;

    public DropCloudFolders(SolomonCluster cluster) {
        this.ydbClient = YdbHelper.createYdbClient(cluster);
        this.root = cluster.kikimrRootPath();

        clustersDao = new YdbClustersDao(ydbClient.table, root + "/Config/V2/Cluster", new ObjectMapper(), ForkJoinPool.commonPool());
        shardsDao = new YdbShardsDao(ydbClient.table, root + "/Config/V2/Shard", new ObjectMapper(), ForkJoinPool.commonPool());
    }

    void run(Path file) throws Exception {
        Table<String, String, Folder> folders = HashBasedTable.create();

        System.err.println("loading shards...");
        for (Shard shard : shardsDao.findAll().join()) {
            Folder folder = folders.get(shard.getProjectId(), shard.getClusterName());
            if (folder == null) {
                folder = new Folder();
                folders.put(shard.getProjectId(), shard.getClusterName(), folder);
            }
            folder.addShard(shard);
        }

        System.err.println("loading clusters...");
        for (Cluster cluster : clustersDao.findAll().join()) {
            Folder folder = folders.get(cluster.getProjectId(), cluster.getName());
            if (folder == null) {
                folder = new Folder();
                folders.put(cluster.getProjectId(), cluster.getName(), folder);
            }
            folder.addCluster(cluster);
        }

        System.err.println("filter shards and clusters to be deleted...");
        Queue<Shard> shardsToDelete = new ConcurrentLinkedQueue<>();
        Queue<Cluster> clustersToDelete = new ConcurrentLinkedQueue<>();
        readClusters(file, (projectId, clusterName) -> {
            Folder folder = folders.get(projectId, clusterName);
            if (folder == null) {
                System.err.printf("skip unknown folder (%s, %s)\n", projectId, clusterName);
                return;
            }

            shardsToDelete.addAll(folder.shards);
            clustersToDelete.addAll(folder.clusters);
        });

        ExecutorService executor = Executors.newFixedThreadPool(THREADS);

        System.err.printf("deleting %d shards... ", shardsToDelete.size());
        CompletableFuture[] futures = IntStream.of(THREADS)
                .mapToObj(id -> CompletableFuture.runAsync(() -> {
                    Shard shard;
                    while ((shard = shardsToDelete.poll()) != null) {
                        shardsDao.deleteOne(shard.getProjectId(), "", shard.getId()).join();
                    }
                }, executor))
                .toArray(CompletableFuture[]::new);
        CompletableFuture.allOf(futures).join();
        System.err.println("done");

        System.err.printf("deleting %d clusters... ", clustersToDelete.size());
        futures = IntStream.of(THREADS)
                .mapToObj(id -> CompletableFuture.runAsync(() -> {
                    Cluster cluster;
                    while ((cluster = clustersToDelete.poll()) != null) {
                        clustersDao.deleteOne(cluster.getProjectId(), "", cluster.getId()).join();
                    }
                }, executor))
                .toArray(CompletableFuture[]::new);
        CompletableFuture.allOf(futures).join();
        System.err.println("done");

        executor.shutdown();
        executor.awaitTermination(5, TimeUnit.SECONDS);
    }

    private static void readClusters(Path file, BiConsumer<String, String> consumer) throws IOException {
        try (var reader = Files.newBufferedReader(file)) {
            reader.readLine(); // skip header

            String line;
            while ((line = reader.readLine()) != null) {
                String[] parts = StringUtils.split(line, '\t');
                String projectId = StringUtils.strip(parts[0]);
                String clusterName = StringUtils.strip(parts[1]);

                consumer.accept(projectId, clusterName);
            }
        }
    }

    private static SolomonCluster cluster(String env) {
        switch (env) {
            case "preprod": return SolomonCluster.CLOUD_PREPROD_FRONT;
            case "prod": return SolomonCluster.CLOUD_PROD_FRONT;
        }
        throw new RuntimeException("unknown env type: " + env);
    }

    @Override
    public void close() {
        ydbClient.close();
    }

    private static final class Folder {
        List<Cluster> clusters = List.of();
        List<Shard> shards = List.of();

        void addCluster(Cluster cluster) {
            if (clusters.isEmpty()) {
                clusters = new ArrayList<>();
            }
            clusters.add(cluster);
        }

        void addShard(Shard shard) {
            if (shards.isEmpty()) {
                shards = new ArrayList<>();
            }
            shards.add(shard);
        }
    }
}
