package ru.yandex.solomon.tool;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.auth.tvm.TvmAuthContext;
import com.yandex.ydb.auth.tvm.YdbClientId;

import ru.yandex.solomon.core.conf.ShardConfMaybeWrong;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.db.dao.ClustersDao;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.core.db.dao.ServicesDao;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbClustersDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbServiceProvidersDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbServicesDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbShardsDao;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.tool.cfg.SolomonCluster;

/**
 * @author Sergey Polovko
 */
public class DropCorruptedConfigs implements AutoCloseable {

    private TvmAuthContext authContext;
    private YdbClient ydbClient;
    private String root;

    private final ServiceProvidersDao serviceProvidersDao;
    private final ProjectsDao projectsDao;
    private final ClustersDao clusterDao;
    private final ServicesDao serviceDao;
    private final ShardsDao shardDao;

    public DropCorruptedConfigs(String env) {
        initYdbClient(env);
        var objectMapper = new ObjectMapper();
        serviceProvidersDao = new YdbServiceProvidersDao(
            ydbClient.table,
            root + "/Config/V2/ServiceProvider",
            objectMapper);
        projectsDao = new YdbProjectsDao(ydbClient.table, root + "/Config/V2/Project", objectMapper, ForkJoinPool.commonPool());
        clusterDao = new YdbClustersDao(ydbClient.table, root + "/Config/V2/Cluster", objectMapper, ForkJoinPool.commonPool());
        serviceDao = new YdbServicesDao(ydbClient.table, root + "/Config/V2/Service", objectMapper, ForkJoinPool.commonPool());
        shardDao = new YdbShardsDao(ydbClient.table, root + "/Config/V2/Shard", objectMapper, ForkJoinPool.commonPool());
    }

    private void initYdbClient(String env) {
        switch (env) {
            case "testing": {
                this.authContext = TvmAuthContext.useTvmApi(2010238, getSelfSecret());
                this.ydbClient = YdbHelper.createYdbClient(
                        "ydb-ru-prestable.yandex.net:2135",
                        "/ru-prestable/solomon/development/solomon",
                        authContext.authProvider(YdbClientId.YDB));
                this.root = "/ru-prestable/solomon/development/solomon";
                break;
            }

            case "prestable": {
                this.authContext = TvmAuthContext.useTvmApi(2010240, getSelfSecret());
                this.ydbClient = YdbHelper.createYdbClient(
                        "ydb-ru-prestable.yandex.net:2135",
                        "/ru-prestable/solomon/prestable/solomon",
                        authContext.authProvider(YdbClientId.YDB));
                this.root = "/ru-prestable/solomon/prestable/solomon";
                break;
            }

            case "production":
                this.ydbClient = YdbHelper.createYdbClient(SolomonCluster.PROD_FRONT);
                this.root = SolomonCluster.PROD_FRONT.kikimrRootPath();
                break;

            case "cloud-preprod":
                this.ydbClient = YdbHelper.createYdbClient(SolomonCluster.CLOUD_PREPROD_FRONT);
                this.root = SolomonCluster.CLOUD_PREPROD_FRONT.kikimrRootPath();
                break;

            case "cloud-prod":
                this.ydbClient = YdbHelper.createYdbClient(SolomonCluster.CLOUD_PROD_FRONT);
                this.root = SolomonCluster.CLOUD_PROD_FRONT.kikimrRootPath();
                break;

            default:
                throw new IllegalStateException("invalid environment type: " + env);
        }
    }

    private static String getSelfSecret() {
        return Objects.requireNonNull(System.getenv("SELF_SECRET"), "please set SELF_SECRET environment variable");
    }

    public static void main(String[] args) {
        if (args.length != 2) {
            System.err.println(
                "Usage: tool {testing|prestable|production|cloud-preprod|cloud-prod} {true|false}\n"
                + "<Specify true/false for dry run>");
            System.exit(1);
        }

        try (var tool = new DropCorruptedConfigs(args[0])) {
            boolean dryRun = Boolean.parseBoolean(args[1]);
            tool.run(dryRun);
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }

        System.exit(0);
    }

    @Override
    public void close() {
        ydbClient.close();

        if (authContext != null) {
            authContext.close();
        }
    }

    void run(boolean dryRun) {
        SolomonRawConf rawConf = loadRaw().join();
        var conf = SolomonConfWithContext.create(rawConf);
        for (ShardConfMaybeWrong shard : conf.getShards()) {
            if (!shard.isCorrect()) {
                System.out.println("drop " + shard.getId() + ", because " + shard.getThrowable().getMessage());
                if (!dryRun) {
                    Shard s = shard.getRaw();
                    shardDao.deleteOne(s.getProjectId(), "", s.getId()).join();
                }
            }
        }
    }

    private CompletableFuture<SolomonRawConf> loadRaw() {
        CompletableFuture<List<ServiceProvider>> serviceProvidersFuture = serviceProvidersDao.findAll();
        CompletableFuture<List<Project>> projectsFuture = projectsDao.findAllNames();
        CompletableFuture<List<Cluster>> clustersFuture = clusterDao.findAll();
        CompletableFuture<List<Service>> servicesFuture = serviceDao.findAll();
        CompletableFuture<List<Shard>> shardsFuture = shardDao.findAll();

        return CompletableFuture.allOf(serviceProvidersFuture, projectsFuture, clustersFuture, servicesFuture, shardsFuture)
                .thenApply(aVoid -> {
                    List<ServiceProvider> serviceProviders = serviceProvidersFuture.getNow(Collections.emptyList());
                    List<Project> projects = projectsFuture.getNow(Collections.emptyList());
                    List<Cluster> clusters = clustersFuture.getNow(Collections.emptyList());
                    List<Service> services = servicesFuture.getNow(Collections.emptyList());
                    List<Shard> shards = shardsFuture.getNow(Collections.emptyList());
                    return new SolomonRawConf(serviceProviders, projects, clusters, services, shards);
                });
    }
}
