package ru.yandex.solomon.tool;

import java.nio.file.Path;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Stopwatch;
import com.google.common.base.Strings;
import io.grpc.Status;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cloud.resourcemanager.GrpcResourceManagerClient;
import ru.yandex.cloud.resourcemanager.ResourceManagerClientOptions;
import ru.yandex.cloud.token.IamTokenClientOptions;
import ru.yandex.cloud.token.Jwt;
import ru.yandex.cloud.token.grpc.GrpcIamTokenClient;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolver;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolverImpl;
import ru.yandex.solomon.core.db.dao.ClustersDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbClustersDao;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.main.logger.LoggerConfigurationUtils;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.async.InFlightLimiter;
import ru.yandex.solomon.util.future.RetryConfig;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static ru.yandex.solomon.util.future.RetryCompletableFuture.runWithRetries;

/**
 * @author Stanislav Kashirin
 */
public final class PatchFoldersInClusters implements AutoCloseable {

    private static final RetryConfig DAO_RETRY_CONFIG = RetryConfig.DEFAULT
        .withNumRetries(10)
        .withDelay(TimeUnit.MILLISECONDS.toMillis(1))
        .withMaxDelay(TimeUnit.MILLISECONDS.toMillis(50));

    private static final RetryConfig RM_RETRY_CONFIG = RetryConfig.DEFAULT
        .withNumRetries(3)
        .withDelay(TimeUnit.MINUTES.toMillis(2))
        .withMaxDelay(TimeUnit.MINUTES.toMillis(5));

    private static final InFlightLimiter inFlightLimiter = new InFlightLimiter(3);

    private static final Duration DEFAULT_CONNECT_TIMEOUT = Duration.ofSeconds(5);
    private static final Duration DEFAULT_REQUEST_TIMEOUT = Duration.ofSeconds(30);

    private final String env;

    private final String folderAlikePrefix;
    private final Stats stats;

    private final Ydb ydb;
    private final ClustersDao clusterDao;

    private final CloudByFolderResolver resolverExistingOnly;
    private final CloudByFolderResolver resolverAll;

    private PatchFoldersInClusters(String env) {
        this.env = env;

        this.folderAlikePrefix = folderAlikePrefix();
        this.stats = new Stats();

        this.ydb = initYdb();
        this.clusterDao = new YdbClustersDao(ydb.client.table, ydb.root + "/Config/V2/Cluster", new ObjectMapper(), ForkJoinPool.commonPool());

        var tokenClient = new GrpcIamTokenClient(newIamTokenClientOptions());
        var jwtBuilder = newJwtBuilder()
            .withPrivateKey(Path.of("/Berkanavt/keys/solomon/iam.pem"))
            .withTtl(Duration.ofHours(1));
        var tokenProvider = TokenProvider.iam(
            tokenClient,
            jwtBuilder,
            Executors.newSingleThreadScheduledExecutor());

        var opts = newResourceManagerClientOptions()
            .withUserAgent("Solomon")
            .withConnectTimeout(DEFAULT_CONNECT_TIMEOUT)
            .withRequestTimeout(DEFAULT_REQUEST_TIMEOUT)
            .withHandlerExecutor(ForkJoinPool.commonPool())
            .withTokenProvider(tokenProvider);
        this.resolverExistingOnly = new CloudByFolderResolverImpl(
            new GrpcResourceManagerClient(opts.withResolveExistingOnly(true)));
        this.resolverAll = new CloudByFolderResolverImpl(
            new GrpcResourceManagerClient(opts.withResolveExistingOnly(false)));
    }

    private Ydb initYdb() {
        switch (env) {
            case "cloud-preprod" -> {
                return new Ydb(
                    YdbHelper.createYdbClient(SolomonCluster.CLOUD_PREPROD_FRONT),
                    SolomonCluster.CLOUD_PREPROD_FRONT.kikimrRootPath());
            }
            case "cloud-prod" -> {
                return new Ydb(
                    YdbHelper.createYdbClient(SolomonCluster.CLOUD_PROD_FRONT),
                    SolomonCluster.CLOUD_PROD_FRONT.kikimrRootPath());
            }
            default -> throw new IllegalStateException("invalid environment type: " + env);
        }
    }

    private IamTokenClientOptions newIamTokenClientOptions() {
        return switch (env) {
            case "cloud-preprod" -> IamTokenClientOptions.forPreprod();
            case "cloud-prod" -> IamTokenClientOptions.forProd();
            default -> throw new IllegalStateException("invalid environment type: " + env);
        };
    }

    private Jwt.Builder newJwtBuilder() {
        return switch (env) {
            case "cloud-preprod" -> Jwt.newBuilder()
                .withAccountId("bfbku4ac1i90nqn3cokr")
                .withKeyId("bfbn3iqodem47j93bm6f");

            case "cloud-prod" -> Jwt.newBuilder()
                .withAccountId("ajed874upg335eded38c")
                .withKeyId("aje5ongk7pbjk88cqhln");

            default -> throw new IllegalStateException("invalid environment type: " + env);
        };
    }

    private ResourceManagerClientOptions newResourceManagerClientOptions() {
        return switch (env) {
            case "cloud-preprod" -> ResourceManagerClientOptions.forPreprod();
            case "cloud-prod" -> ResourceManagerClientOptions.forProd();
            default -> throw new IllegalStateException("invalid environment type: " + env);
        };
    }

    private String folderAlikePrefix() {
        return switch (env) {
            case "cloud-preprod" -> "aoe";
            case "cloud-prod" -> "b1g";
            default -> throw new IllegalStateException("invalid environment type: " + env);
        };
    }

    public static void main(String[] args) {
        LoggerConfigurationUtils.disableLogger();

        if (args.length != 2) {
            System.err.println(
                "Usage: tool {cloud-preprod|cloud-prod} {true|false}\n"
                    + "<Specify true/false for dry run>");
            System.exit(1);
        }

        try (var tool = new PatchFoldersInClusters(args[0])) {
            boolean dryRun = Boolean.parseBoolean(args[1]);
            tool.run(dryRun);
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }

        System.exit(0);
    }

    @Override
    public void close() {
        ydb.client.close();
    }

    private void run(boolean dryRun) {
        System.out.printf("patching folders in clusters: env=%s dryRun=%s%n", env, dryRun);
        var sw = Stopwatch.createStarted();

        var resolvedClusters = resolveClusters().join();
        System.out.println("clusters resolved: " + resolvedClusters.size());

        int counter = 0;
        for (var resolvedCluster : resolvedClusters) {
            if (resolvedCluster instanceof VanillaCluster vanillaCluster) {
                processVanillaCluster(vanillaCluster, dryRun);
            } else if (resolvedCluster instanceof ClusterWithFolder clusterWithFolder) {
                processClusterWithFolder(clusterWithFolder, dryRun);
            }

            if (++counter % 1000 == 0) {
                System.err.printf("progress: %.2f%%%n", counter * 100.0 / resolvedClusters.size());
            }
        }

        System.out.println("total clusters: " + resolvedClusters.size());
        System.out.println("vanilla clusters: " + stats.vanillaCount);
        System.out.println("folder clusters: " + stats.folderCount);
        System.out.println("folder clusters (existing folder): " + stats.existingFolderCount);
        System.out.println("----");

        System.out.println("suspicious vanilla distribution: ");
        System.out.println(prettify(stats.suspiciousVanillaDistribution));
        System.out.println("----");

        System.out.println("non-existing folder distribution: ");
        System.out.println(prettify(stats.nonExistingFolderDistribution));
        System.out.println("----");

        System.out.println("elapsed: " + sw.elapsed());
    }

    private void processVanillaCluster(VanillaCluster vanillaCluster, boolean dryRun) {
        var cluster = vanillaCluster.cluster();

        stats.vanillaCount++;
        var clusterName = cluster.getName();
        if (clusterName.startsWith(folderAlikePrefix) || clusterName.startsWith("yc.")) {
            System.out.printf("suspicious vanilla: clusterId=%s%n", cluster.getId());
            stats.suspiciousVanillaDistribution.addTo(cluster.getProjectId(), 1);
        }

        var folderId = Strings.nullToEmpty(cluster.getFolderId());
        if (!folderId.isEmpty()) {
            System.out.printf(
                "drop unknown folderId: clusterId=%s folderId=%s%n",
                cluster.getId(),
                folderId);

            if (!dryRun) {
                updateCluster(cluster, "").join();
            }
        }
    }

    private void processClusterWithFolder(ClusterWithFolder clusterWithFolder, boolean dryRun) {
        var cluster = clusterWithFolder.cluster();

        stats.folderCount++;
        if (clusterWithFolder.folderExists()) {
            stats.existingFolderCount++;
        } else {
            stats.nonExistingFolderDistribution.addTo(cluster.getProjectId(), 1);
        }

        if (!cluster.getProjectId().equals(clusterWithFolder.resolvedCloudId())) {
            System.out.printf(
                "projectId/resolvedCloudId mismatch: clusterId=%s projectId=%s resolvedCloudId=%s%n",
                cluster.getId(),
                cluster.getProjectId(),
                clusterWithFolder.resolvedCloudId());

            if (dryRun) {
                if (Boolean.parseBoolean(System.getProperty("failFast"))) {
                    throw new IllegalStateException(
                        "projectId/resolvedCloudId mismatch: clusterId=%s projectId=%s resolvedCloudId=%s"
                            .formatted(
                                cluster.getId(),
                                cluster.getProjectId(),
                                clusterWithFolder.resolvedCloudId()));
                }
            }
        }

        if (!clusterWithFolder.resolvedFolderId().equals(cluster.getFolderId())) {
            System.out.printf(
                "change folderId: clusterId=%s folderId=%s prev=%s%n",
                cluster.getId(),
                clusterWithFolder.resolvedFolderId(),
                cluster.getFolderId());

            if (!dryRun) {
                updateCluster(cluster, clusterWithFolder.resolvedFolderId()).join();
            }
        }
    }

    private CompletableFuture<Cluster> updateCluster(Cluster cluster, String newFolderId) {
        var newCluster = cluster.toBuilder()
            .setFolderId(newFolderId)
            .build();

        return clusterDao.partialUpdate(newCluster)
            .thenCompose(
                updated -> updated
                    .map(CompletableFuture::completedFuture)
                    .orElseGet(
                        () -> runWithRetries(
                            () -> tryUpdateCluster(cluster.getProjectId(), cluster.getId(), newFolderId),
                            DAO_RETRY_CONFIG)));
    }

    private CompletableFuture<Cluster> tryUpdateCluster(
        String projectId,
        String clusterId,
        String newFolderId)
    {
        return clusterDao.findOne(projectId, "", clusterId)
            .thenCompose(cluster -> {
                if (cluster.isEmpty()) {
                    System.out.printf("meanwhile, cluster is gone: clusterId=%s%n", clusterId);
                    return completedFuture(null);
                }

                var newCluster = cluster.orElseThrow().toBuilder()
                    .setFolderId(newFolderId)
                    .build();

                return clusterDao.partialUpdate(newCluster)
                    .thenApply(
                        updated -> updated.orElseThrow(
                            () -> Status.ABORTED
                                .withDescription("unable to update cluster")
                                .asRuntimeException()));
            });
    }

    private CompletableFuture<? extends List<ResolvedCluster>> resolveClusters() {
        var resolveEmptyFolderOnly = Boolean.getBoolean("resolveEmptyFolderOnly");
        System.out.println("resolveEmptyFolderOnly: " + resolveEmptyFolderOnly);

        return clusterDao.findAll()
            .thenCompose(
                clusters -> clusters.stream()
                    .filter(cluster -> {
                        if (resolveEmptyFolderOnly) {
                            var folderId = Strings.nullToEmpty(cluster.getFolderId());
                            return folderId.isEmpty();
                        }

                        return true;
                    })
                    .map(this::resolveCluster)
                    .collect(
                        collectingAndThen(
                            toList(),
                            CompletableFutures::allOf)));
    }

    private CompletableFuture<ResolvedCluster> resolveCluster(Cluster cluster) {
        var resultFuture = new CompletableFuture<ResolvedCluster>();
        inFlightLimiter.run(() -> resolveCluster(cluster, resultFuture));
        return resultFuture;
    }

    private CompletableFuture<?> resolveCluster(
        Cluster cluster,
        CompletableFuture<ResolvedCluster> resultFuture)
    {
        var clusterName = cluster.getName();

        return tryResolveAll(clusterName)
            .whenComplete((resolvedCloud, thr) -> {
                if (thr != null) {
                    resultFuture.completeExceptionally(thr);
                } else {
                    if (resolvedCloud.isEmpty()) {
                        resultFuture.complete(new VanillaCluster(cluster));
                    } else {
                        tryResolveExistingOnly(clusterName)
                            .whenComplete((existingResolvedCloud, t) -> {
                                if (t != null) {
                                    resultFuture.completeExceptionally(t);
                                } else {
                                    resultFuture.complete(
                                        new ClusterWithFolder(
                                            cluster,
                                            clusterName,
                                            resolvedCloud.get(),
                                            existingResolvedCloud.isPresent()));
                                }
                            });
                    }
                }
            });
    }

    private CompletableFuture<Optional<String>> tryResolveAll(String clusterName) {
        return runWithRetries(() -> resolverAll.tryResolveCloudId(clusterName), RM_RETRY_CONFIG);
    }

    private CompletableFuture<Optional<String>> tryResolveExistingOnly(String clusterName) {
        return runWithRetries(() -> resolverExistingOnly.tryResolveCloudId(clusterName), RM_RETRY_CONFIG);
    }

    private static String prettify(Map<String, Integer> distribution) {
        return distribution.entrySet().stream()
            .sorted(Map.Entry.<String, Integer>comparingByValue().reversed())
            .map(e -> e.getKey() + ": " + e.getValue())
            .collect(joining("\n"));
    }

    private class Stats {
        int vanillaCount, folderCount, existingFolderCount;
        final Object2IntOpenHashMap<String> suspiciousVanillaDistribution = new Object2IntOpenHashMap<>();
        final Object2IntOpenHashMap<String> nonExistingFolderDistribution = new Object2IntOpenHashMap<>();
    }

    private record Ydb(YdbClient client, String root) {}

    @SuppressWarnings("MarkerInterface")
    sealed interface ResolvedCluster {}

    @ParametersAreNonnullByDefault
    private record VanillaCluster(Cluster cluster)
        implements ResolvedCluster {}

    @ParametersAreNonnullByDefault
    private record ClusterWithFolder(
        Cluster cluster,
        String resolvedFolderId,
        String resolvedCloudId,
        boolean folderExists) implements ResolvedCluster {
    }

}
