package ru.yandex.solomon.tool;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.MapDifference;
import com.google.common.collect.Maps;
import com.google.common.collect.Table;
import com.google.common.collect.Tables;
import com.yandex.ydb.table.TableClient;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.thread.StubThreadPoolProvider;
import ru.yandex.solomon.core.conf.ShardNumIdGenerator;
import ru.yandex.solomon.core.conf.ShardNumIdGeneratorImpl;
import ru.yandex.solomon.core.db.dao.ClustersDao;
import ru.yandex.solomon.core.db.dao.ConfigDaoContext;
import ru.yandex.solomon.core.db.dao.DashboardsDao;
import ru.yandex.solomon.core.db.dao.GraphsDao;
import ru.yandex.solomon.core.db.dao.ProjectMenuDao;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ServicesDao;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Dashboard;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.graph.Graph;
import ru.yandex.solomon.flags.FeatureFlagHolderStub;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class ProjectConfigsSync implements AutoCloseable {
    private static final int MAX_CHANGE_IN_FLIGHT = 100;

    private final ExecutorService executor;
    private final Unit from;
    private final Unit to;

    private ProjectConfigsSync(Unit from, Unit to) {
        this.from = from;
        this.to = to;
        this.executor = Executors.newFixedThreadPool(4);
    }

    public static void main(String[] args) throws InterruptedException {
        while (true) {
            try (ProjectConfigsSync sync = new ProjectConfigsSync(Unit.productionUnit(), Unit.prestableUnit())) {
                try {
                    var projects = Unit.productionUnit().projectsDao.findAllNames().join()
                        .stream()
                        .filter(project -> project.getId().equals("solomon"))
                        .collect(Collectors.toList());

                    AtomicInteger cursor = new AtomicInteger();
                    AsyncActorBody body = () -> {
                        var pos = cursor.getAndIncrement();
                        if (pos >= projects.size()) {
                            return completedFuture(AsyncActorBody.DONE_MARKER);
                        }

                        var project = projects.get(pos);
                        return sync.to.projectsDao.findById(project.getId())
                            .thenCompose(opt -> {
                                if (opt.isPresent()) {
                                    return CompletableFuture.completedFuture(null);
                                } else {
                                    return sync.to.projectsDao.insert(project);
                                }
                            })
                            .thenCompose(ignore -> sync.sync(project.getId(), SyncOptions.INSERT_AND_UPDATE));
                    };

                    AsyncActorRunner runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 1);
                    runner.start().join();
                    System.exit(0);
                } catch (Throwable e) {
                    e.printStackTrace();
                    TimeUnit.SECONDS.sleep(30);
                }
            }
        }

//        try (ProjectConfigsSync sync = new ProjectConfigsSync(Unit.productionUnit(), Unit.prestableUnit())) {
//            sync.sync("solomon", SyncOptions.INSERT_AND_UPDATE).join();
//            System.exit(0);
//        }
    }

    private CompletableFuture<Void> sync(String projectId, SyncOptions options) {
        System.out.println(projectId + " starting...");
        return CompletableFuture.completedFuture(null)
            .thenCompose(ignore -> syncClusters(projectId, options))
            .thenCompose(ignore -> syncServices(projectId, options))
            .thenCompose(ignore -> syncShards(projectId, options))
            //.thenCompose(ignore -> syncGraphs(projectId, options))
            //.thenCompose(ignore -> syncDashboards(projectId, options))
            //.thenCompose(ignore -> syncMenu(projectId, options))
            .whenComplete((ignore, e) -> {
                if (e == null) {
                    System.out.println(projectId + " done");
                } else {
                    System.err.println(projectId + " failed");
                    e.printStackTrace();
                }
            });
    }

    private CompletableFuture<Void> syncClusters(String projectId, SyncOptions options) {
        return CompletableFutures.allOf2(getClusters(from, projectId), getClusters(to, projectId))
            .thenCompose(tuple -> {
                MapDifference<String, Cluster> diff = Maps.difference(tuple.get1(), tuple.get2());

                List<Supplier<CompletableFuture<?>>> tasks = new ArrayList<>();

                if (options.canInsert) {
                    for (Cluster cluster : diff.entriesOnlyOnLeft().values()) {
                        tasks.add(() -> {
                            System.out.println(projectId + " insert cluster: " + cluster.getId());
                            return to.daoClusters.insert(cluster);
                        });
                    }
                }

                if (options.canDelete) {
                    for (Cluster cluster : diff.entriesOnlyOnRight().values()) {
                        tasks.add(() -> {
                            System.out.println(projectId + " delete cluster: " + cluster.getId());
                            return to.daoClusters.deleteOne(projectId, "", cluster.getId());
                        });
                    }
                }

                if (options.canUpdate) {
                    for (MapDifference.ValueDifference<Cluster> cluster : diff.entriesDiffering().values()) {
                        var update = cluster.leftValue().toBuilder()
                            .setVersion(cluster.rightValue().getVersion())
                            .build();

                        if (update.equals(cluster.rightValue())) {
                            continue;
                        }

                        tasks.add(() -> {
                            System.out.println(projectId + " update cluster: " + update.getId());
                            return to.daoClusters.partialUpdate(update);
                        });
                    }
                }

                return applyChanges(tasks);
            });
    }

    private CompletableFuture<Void> syncServices(String projectId, SyncOptions options) {
        return CompletableFutures.allOf2(getServices(from, projectId), getServices(to, projectId))
            .thenCompose(tuple -> {
                MapDifference<String, Service> diff = Maps.difference(tuple.get1(), tuple.get2());

                List<Supplier<CompletableFuture<?>>> tasks = new ArrayList<>();

                if (options.canInsert) {
                    for (Service service : diff.entriesOnlyOnLeft().values()) {
                        tasks.add(() -> {
                            System.out.println(projectId + " insert service: " + service.getId());
                            return to.daoServices.insert(service);
                        });
                    }
                }

                if (options.canDelete) {
                    for (Service service : diff.entriesOnlyOnRight().values()) {
                        tasks.add(() -> {
                            System.out.println(projectId + " delete service: " + service.getId());
                            return to.daoServices.deleteOne(projectId, "", service.getId());
                        });
                    }
                }

                if (options.canUpdate) {
                    for (MapDifference.ValueDifference<Service> service : diff.entriesDiffering().values()) {
                        var update = service.leftValue()
                            .toBuilder()
                            .setVersion(service.rightValue().getVersion())
                            .build();

                        if (update.equals(service.rightValue())) {
                            continue;
                        }

                        tasks.add(() -> {
                            System.out.println(projectId + " update service: " + update.getId());
                            return to.daoServices.partialUpdate(update);
                        });
                    }
                }

                return applyChanges(tasks);
            });
    }

    private CompletableFuture<Void> syncShards(String projectId, SyncOptions options) {
        return CompletableFutures.allOf2(getShards(from, projectId), getShards(to, projectId))
            .thenCompose(tuple -> {
                MapDifference<String, Shard> diff = Maps.difference(tuple.get1(), tuple.get2());

                List<Supplier<CompletableFuture<?>>> tasks = new ArrayList<>();

                if (options.canInsert) {
                    for (Shard shard : diff.entriesOnlyOnLeft().values()) {
                        tasks.add(() -> {
                            System.out.println(projectId + " insert shard: " + shard.getId());
                            return to.daoShards.insert(shard)
                                .thenCompose(success -> {
                                    if (!success) {
                                        return to.daoShards.insert(shard.toBuilder()
                                            .setNumId(to.numIdGenerator.generateNumId(shard.getId()))
                                            .build())
                                            .whenComplete((aBoolean, throwable) -> {
                                                System.out.println(projectId + " inserted shard: " + shard.getId() + "  " + aBoolean);
                                            });
                                    }
                                    return completedFuture(null);
                                });
                        });
                    }
                }

                if (options.canDelete) {
                    for (Shard shard : diff.entriesOnlyOnRight().values()) {
                        tasks.add(() -> {
                            System.out.println(projectId + " delete shard: " + shard.getId());
                            return to.daoShards.deleteOne(projectId, "", shard.getId());
                        });
                    }
                }

                if (options.canUpdate) {
                    for (MapDifference.ValueDifference<Shard> shard : diff.entriesDiffering().values()) {
                        Shard update = shard.leftValue()
                            .toBuilder()
                            .setVersion(shard.rightValue().getVersion())
                            .setNumId(shard.rightValue().getNumId())
                            .build();

                        if (update.equals(shard.rightValue())) {
                            continue;
                        }

                        tasks.add(() -> {
                            System.out.println(projectId + " update shard: " + shard.leftValue().getId());
                            return to.daoShards.partialUpdate(update, true);
                        });
                    }
                }

                return applyChanges(tasks);
            });
    }

    private CompletableFuture<Void> syncGraphs(String projectId, SyncOptions options) {
        return CompletableFutures.allOf2(getGraphs(from, projectId), getGraphs(to, projectId))
            .thenCompose(tuple -> {
                MapDifference<String, Graph> diff = Maps.difference(tuple.get1(), tuple.get2());

                List<Supplier<CompletableFuture<?>>> tasks = new ArrayList<>();

                if (options.canInsert) {
                    for (Graph graph : diff.entriesOnlyOnLeft().values()) {
                        tasks.add(() -> {
                            System.out.println(projectId + " insert graph: " + graph.getId());
                            return to.daoGraphs.insert(graph);
                        });
                    }
                }

                if (options.canDelete) {
                    for (Graph graph : diff.entriesOnlyOnRight().values()) {
                        tasks.add(() -> {
                            System.out.println(projectId + " delete graph: " + graph.getId());
                            return to.daoGraphs.deleteOne(projectId, "", graph.getId());
                        });
                    }
                }

                if (options.canUpdate) {
                    for (MapDifference.ValueDifference<Graph> graph : diff.entriesDiffering().values()) {
                        var update = graph.leftValue()
                            .toBuilder()
                            .setVersion(graph.rightValue().getVersion())
                            .build();

                        if (update.equals(graph.rightValue())) {
                            continue;
                        }

                        tasks.add(() -> {
                            System.out.println(projectId + " update graph: " + update.getId());
                            return to.daoGraphs.partialUpdate(update);
                        });
                    }
                }

                return applyChanges(tasks);
            });
    }

    private CompletableFuture<Void> syncDashboards(String projectId, SyncOptions options) {
        return CompletableFutures.allOf2(getDashboards(from, projectId), getDashboards(to, projectId))
            .thenCompose(tuple -> {
                MapDifference<String, Dashboard> diff = Maps.difference(tuple.get1(), tuple.get2());

                List<Supplier<CompletableFuture<?>>> tasks = new ArrayList<>();
                if (options.canInsert) {
                    for (Dashboard dashboard : diff.entriesOnlyOnLeft().values()) {
                        tasks.add(() -> {
                            System.out.println(projectId + " insert dashboard: " + dashboard.getId());
                            return to.daoDashboards.insert(dashboard);
                        });
                    }
                }

                if (options.canDelete) {
                    for (Dashboard dashboard : diff.entriesOnlyOnRight().values()) {
                        tasks.add(() -> {
                            System.out.println(projectId + " delete dashboard: " + dashboard.getId());
                            return to.daoDashboards.deleteOne(projectId, "", dashboard.getId());
                        });
                    }
                }

                if (options.canUpdate) {
                    for (MapDifference.ValueDifference<Dashboard> dashboard : diff.entriesDiffering().values()) {
                        var update = dashboard.leftValue()
                            .toBuilder()
                            .setVersion(dashboard.rightValue().getVersion())
                            .build();

                        if (update.equals(dashboard.rightValue())) {
                            continue;
                        }

                        tasks.add(() -> {
                            System.out.println(projectId + " update dashboard: " + update.getId());
                            return to.daoDashboards.partialUpdate(update);
                        });
                    }
                }

                return applyChanges(tasks);
            });
    }

    private CompletableFuture<Void> syncMenu(String projectId, SyncOptions options) {
        return CompletableFutures.allOf2(from.daoMenu.findById(projectId), to.daoMenu.findById(projectId))
            .thenCompose(tuple -> {
                if (options.canUpdate && tuple.get1().isPresent()) {
                    System.out.println(projectId + " update menu");
                    return to.daoMenu.deleteById(projectId)
                        .thenCompose(ignore -> to.daoMenu.upsert(tuple.get1().get()))
                        .thenApply(ignore -> null);
                } else if (options.canDelete && tuple.get2().isPresent()) {
                    System.out.println(projectId + " delete menu");
                    return to.daoMenu.deleteById(projectId);
                } else {
                    return completedFuture(null);
                }
            });
    }

    private CompletableFuture<Void> applyChanges(List<Supplier<CompletableFuture<?>>> tasks) {
        if (tasks.isEmpty()) {
            return completedFuture(null);
        }

        AtomicInteger index = new AtomicInteger();
        AsyncActorBody body = () -> {
            int i = index.getAndIncrement();
            if (i >= tasks.size()) {
                return completedFuture(AsyncActorBody.DONE_MARKER);
            }

            return tasks.get(i).get();
        };

        return new AsyncActorRunner(body, executor, MAX_CHANGE_IN_FLIGHT).start();
    }

    private CompletableFuture<Map<String, Cluster>> getClusters(Unit unit, String projectId) {
        if (unit.clusters == null) {
            return unit.daoClusters.findAll()
                .thenCompose(clusters -> {
                    unit.clusters = clusters.stream()
                        .collect(Tables.toTable(Cluster::getProjectId, Cluster::getId, Function.identity(), HashBasedTable::create));
                    return getClusters(unit, projectId);
                });
        }

        return completedFuture(Objects.requireNonNull(unit.clusters).row(projectId));
    }

    private CompletableFuture<Map<String, Service>> getServices(Unit unit, String projectId) {
        if (unit.services == null) {
            return unit.daoServices.findAll()
                .thenCompose(services -> {
                    unit.services = services.stream()
                        .collect(Tables.toTable(Service::getProjectId, Service::getId, Function.identity(), HashBasedTable::create));
                    return getServices(unit, projectId);
                });
        }

        return completedFuture(Objects.requireNonNull(unit.services).row(projectId));
    }

    private CompletableFuture<Map<String, Shard>> getShards(Unit unit, String projectId) {
        if (unit.shards == null) {
            return unit.daoShards.findAll()
                .thenCompose(shards -> {
                    unit.shards = shards.stream()
                        .collect(Tables.toTable(Shard::getProjectId, Shard::getId, Function.identity(), HashBasedTable::create));
                    return getShards(unit, projectId);
                });
        }

        return completedFuture(Objects.requireNonNull(unit.shards).row(projectId));
    }

    private CompletableFuture<Map<String, Graph>> getGraphs(Unit unit, String projectId) {
        if (unit.graphs == null) {
            return unit.daoGraphs.findAll()
                .thenCompose(graphs -> {
                    unit.graphs = graphs.stream()
                        .collect(Tables.toTable(Graph::getProjectId, Graph::getId, Function.identity(), HashBasedTable::create));
                    return getGraphs(unit, projectId);
                });
        }

        return completedFuture(Objects.requireNonNull(unit.graphs).row(projectId));
    }

    private CompletableFuture<Map<String, Dashboard>> getDashboards(Unit unit, String projectId) {
        if (unit.dashboards == null) {
            return unit.daoDashboards.findAll()
                .thenCompose(dashboards -> {
                    unit.dashboards = dashboards.stream()
                        .collect(Tables.toTable(Dashboard::getProjectId, Dashboard::getId, Function.identity(), HashBasedTable::create));
                    return getDashboards(unit, projectId);
                });
        }

        return completedFuture(Objects.requireNonNull(unit.dashboards).row(projectId));
    }

    @Override
    public void close() {
        from.close();
        to.close();
        executor.shutdownNow();
    }

    private static class Unit implements AutoCloseable {

        private static Unit productionUnit() {
            return new Unit(
                SolomonCluster.PROD_FRONT
            );
        }

        private static Unit prestableUnit() {
            return new Unit(
                SolomonCluster.PRESTABLE_FRONT
            );
        }

        private static Unit testingUnit() {
            return new Unit(
                SolomonCluster.TEST_FRONT
            );
        }

        final SolomonCluster ydbCluster;
        final TableClient tableClient;
        final YdbClient ydb;
        final ProjectsDao projectsDao;
        final ClustersDao daoClusters;
        final ServicesDao daoServices;
        final ShardsDao daoShards;
        final ShardNumIdGenerator numIdGenerator;
        final GraphsDao daoGraphs;
        final DashboardsDao daoDashboards;
        final ProjectMenuDao daoMenu;

        @Nullable
        volatile Table<String, String, Shard> shards;
        @Nullable
        volatile Table<String, String, Cluster> clusters;
        @Nullable
        volatile Table<String, String, Service> services;
        @Nullable
        volatile Table<String, String, Graph> graphs;
        @Nullable
        volatile Table<String, String, Dashboard> dashboards;

        private Unit(SolomonCluster ydbCluster) {
            this.ydbCluster = ydbCluster;
            this.ydb = YdbHelper.createYdbClient(ydbCluster);
            this.tableClient = ydb.table;

            var daoCtx = new ConfigDaoContext(tableClient, ydbCluster.kikimrRootPath(), null, new FeatureFlagHolderStub(), new StubThreadPoolProvider());
            this.projectsDao = daoCtx.projectsDao(Optional.empty());
            this.daoClusters = daoCtx.clustersDao();
            this.daoServices = daoCtx.servicesDao();
            this.daoShards = daoCtx.shardsDao();
            this.daoGraphs = daoCtx.graphsDao();
            this.daoDashboards = daoCtx.dashboardsDao();
            this.daoMenu = daoCtx.projectMenuDao();
            this.numIdGenerator = new ShardNumIdGeneratorImpl(daoShards, ForkJoinPool.commonPool(), Executors.newSingleThreadScheduledExecutor(), new MetricRegistry());
        }

        @Override
        public void close() {
            tableClient.close();
        }
    }

    private static class SyncOptions {

        // Full synchronization
        private static final SyncOptions FULL = new SyncOptions(true, true, true);
        // Partial synchronization - don't delete waste entities from target for history
        private static final SyncOptions INSERT_AND_UPDATE = new SyncOptions(true, true, false);
        // Options to insert new entities from source only
        private static final SyncOptions INSERT_ONLY = new SyncOptions(true, false, false);

        private final boolean canInsert;
        private final boolean canUpdate;
        private final boolean canDelete;

        SyncOptions(boolean canInsert, boolean canUpdate, boolean canDelete) {
            this.canInsert = canInsert;
            this.canUpdate = canUpdate;
            this.canDelete = canDelete;
        }
    }
}
