package ru.yandex.solomon.tool;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

import javax.annotation.WillCloseWhenClosed;

import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.rpc.RpcTransport;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.rpc.grpc.GrpcSchemeRpc;
import com.yandex.ydb.table.rpc.grpc.GrpcTableRpc;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.transaction.TxControl;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbClient implements AutoCloseable {
    @WillCloseWhenClosed
    private final RpcTransport transport;
    public final SchemeClient scheme;
    public final TableClient table;

    public YdbClient(@WillCloseWhenClosed RpcTransport transport) {
        this.transport = transport;
        this.scheme = SchemeClient.newClient(GrpcSchemeRpc.useTransport(transport)).build();
        this.table = TableClient.newClient(GrpcTableRpc.useTransport(transport))
            .sessionPoolSize(50, 1000)
            .sessionCreationMaxRetries(100)
            .build();
    }

    public Fluent fluent() {
        return new Fluent();
    }

    @Override
    public void close() {
        scheme.close();
        table.close();
        transport.close();
    }

    public class Fluent {
        private SessionRetryContext.Builder retry;
        private TxControl tx = TxControl.serializableRw().setCommitTx(true);
        private ExecuteDataQuerySettings settings = new ExecuteDataQuerySettings()
            .keepInQueryCache();

        private Fluent() {
            retry = SessionRetryContext.create(table);
        }

        public Fluent retryForever() {
            retry.maxRetries(Integer.MAX_VALUE);
            return this;
        }

        public Fluent tx(TxControl tx) {
            this.tx = tx;
            return this;
        }

        public Fluent timeout(long value, TimeUnit unit) {
            settings.setTimeout(value, unit);
            return this;
        }

        public CompletableFuture<Result<DataQueryResult>> execute(String query) {
            return execute(query, Params.empty());
        }

        public CompletableFuture<Result<DataQueryResult>> execute(String query, Params params) {
            return retry.build().supplyResult(session -> session.executeDataQuery(query, tx, params, settings));
        }

        public CompletableFuture<Status> createTable(String path, TableDescription tableDescriptions) {
            return retry.build().supplyStatus(session -> session.createTable(path, tableDescriptions));
        }

        public CompletableFuture<Status> executeOnSession(Function<Session, CompletableFuture<Status>> fn) {
            return retry.build().supplyStatus(fn);
        }


        public <T> CompletableFuture<Result<T>> callOnSession(Function<Session, CompletableFuture<Result<T>>> fn) {
            return retry.build().supplyResult(fn);
        }

        public CompletableFuture<Result<TableDescription>> describeTable(String path) {
            return retry.build().supplyResult(session -> session.describeTable(path));
        }
    }
}
