package ru.yandex.solomon.tool;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import com.google.common.net.HostAndPort;
import com.yandex.ydb.core.auth.AuthProvider;
import com.yandex.ydb.core.auth.NopAuthProvider;
import com.yandex.ydb.core.auth.TokenAuthProvider;
import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.core.rpc.RpcTransport;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.rpc.grpc.GrpcSchemeRpc;
import com.yandex.ydb.table.rpc.grpc.GrpcTableRpc;
import io.netty.buffer.ByteBufAllocator;
import io.netty.channel.ChannelOption;
import io.netty.channel.EventLoopGroup;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cloud.token.IamTokenClientOptions;
import ru.yandex.cloud.token.Jwt;
import ru.yandex.cloud.token.grpc.GrpcIamTokenClient;
import ru.yandex.misc.thread.factory.DaemonThreadFactory;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.NettyUtils;
import ru.yandex.solomon.ydb.YdbAuthProviders.IamAuthProvider;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbHelper {
    public static YdbClient createYdbClient(SolomonCluster cluster) {
        return new YdbClient(createRpcTransport(cluster));
    }

    public static YdbClient createYdbClient(HostAndPort host) {
        return createYdbClient(List.of(host));
    }

    public static YdbClient createYdbClient(String endpoint, String database, AuthProvider auth) {
        var transport = createRpcTransport(endpoint, database, auth);
        return new YdbClient(transport);
    }

    public static YdbClient createYdbClient(List<HostAndPort> hosts) {
        var transport = createRpcTransport(hosts);
        return new YdbClient(transport);
    }

    public static TableClient createTableClient(SolomonCluster cluster) {
        return createTableClient(createRpcTransport(cluster));
    }

    public static SchemeClient createSchemaClient(SolomonCluster cluster) {
        return createSchemaClient(createRpcTransport(cluster));
    }

    public static TableClient createTableClient(RpcTransport transport) {
        return TableClient.newClient(GrpcTableRpc.ownTransport(transport))
            .sessionPoolSize(10, 100)
            .sessionCreationMaxRetries(Integer.MAX_VALUE)
            .build();
    }

    public static SchemeClient createSchemaClient(RpcTransport transport) {
        return SchemeClient.newClient(GrpcSchemeRpc.ownTransport(transport)).build();
    }

    // XXX: the source of truth is solomon/configs. We just need to learn how to access them from code without messing
    // with arcadia dir or cwd
    public static RpcTransport createRpcTransport(SolomonCluster cluster) {
        var builder = switch (cluster) {
            case PRESTABLE_FRONT -> fillDefaultOpts(
                    GrpcTransport.forEndpoint("ydb-ru-prestable.yandex.net:2135", "/ru-prestable/solomon/prestable/solomon"));
            case TEST_FRONT -> fillDefaultOpts(
                    GrpcTransport.forEndpoint("ydb-ru-prestable.yandex.net:2135", "/ru-prestable/solomon/development/solomon"));
            case CLOUD_PROD_FRONT -> fillDefaultOpts(
                    GrpcTransport.forEndpoint("solomon-dn.ydb.cloud.yandex.net:2136", "/global/solomon"))
                    .withSecureConnection();
            case CLOUD_PREPROD_FRONT -> fillDefaultOpts(
                    GrpcTransport.forEndpoint("solomon-dn.ydb.cloud-preprod.yandex.net:2136", "/pre-prod_global/solomon"))
                    .withSecureConnection();
            default -> fillDefaultOpts(GrpcTransport.forHosts(cluster.addressesKikimrGrpc()));
        };
        return builder
                .withAuthProvider(createAuthProvider(cluster))
                .build();
    }

    public static RpcTransport createRpcTransport(List<HostAndPort> hosts) {
        return fillDefaultOpts(GrpcTransport.forHosts(hosts)).build();
    }

    public static RpcTransport createRpcTransport(String endpoint, String database, AuthProvider auth) {
        return fillDefaultOpts(GrpcTransport.forEndpoint(endpoint, database)
            .withAuthProvider(auth))
            .build();
    }

    public static RpcTransport createSecureRpcTransport(String endpoint, String database, AuthProvider auth) {
        return fillDefaultOpts(GrpcTransport.forEndpoint(endpoint, database)
                .withSecureConnection()
                .withAuthProvider(auth))
                .build();
    }

    public static AuthProvider createAuthProvider(SolomonCluster cluster) {
        switch (cluster) {
            case TEST_FRONT:
            case PRESTABLE_FRONT: {
                try {
                    return new TokenAuthProvider(Files.readString(Path.of(
                            System.getenv("HOME"), ".ydb/token")).strip());
                } catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
            case CLOUD_PREPROD_FRONT: {
                var jwtBuilder = Jwt.newBuilder()
                        .withAccountId("yc.monitoring.ydb-sa")
                        .withKeyId("bfb3f80ur7v826dnjcql")
                        .withTtl(Duration.ofHours(1))
                        .withPrivateKey(Path.of("/Berkanavt/keys/solomon/ydb_global_iam.pem"));

                var timer = Executors.newScheduledThreadPool(1, new DaemonThreadFactory());
                var iamClient = new GrpcIamTokenClient(IamTokenClientOptions.forPreprod());
                return new IamAuthProvider(TokenProvider.iam(iamClient, jwtBuilder, timer));
            }
            case CLOUD_PROD_FRONT: {
                var jwtBuilder = Jwt.newBuilder()
                        .withAccountId("yc.monitoring.ydb-sa")
                        .withKeyId("aje53oiheeja654qtha3")
                        .withTtl(Duration.ofHours(1))
                        .withPrivateKey(Path.of("/Berkanavt/keys/solomon/ydb_global_iam.pem"));

                var timer = Executors.newScheduledThreadPool(1, new DaemonThreadFactory());
                var iamClient = new GrpcIamTokenClient(IamTokenClientOptions.forProd());
                return new IamAuthProvider(TokenProvider.iam(iamClient, jwtBuilder, timer));
            }
            default:
                return NopAuthProvider.INSTANCE;
        }
    }

    public static GrpcTransport.Builder fillDefaultOpts(GrpcTransport.Builder builder) {
        final EventLoopGroup ioExecutor = NettyUtils.createEventLoopGroup("io", Runtime.getRuntime().availableProcessors());
        final ExecutorService callExecutor = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors());

        final int connectTimeoutMillis = Math.toIntExact(TimeUnit.SECONDS.toMillis(5));
        final int maxMessageSize = 256 << 20;
        final Duration readTimeout = Duration.ofMinutes(1);

        return builder.withReadTimeout(readTimeout)
            .withCallExecutor(callExecutor)
            .withChannelInitializer(channel -> {
                channel.channelType(NettyUtils.clientChannelTypeForEventLoop(ioExecutor));
                channel.eventLoopGroup(ioExecutor);
                channel.maxInboundMessageSize(maxMessageSize);
                channel.withOption(ChannelOption.ALLOCATOR, ByteBufAllocator.DEFAULT);
                channel.withOption(ChannelOption.TCP_NODELAY, Boolean.TRUE);
                channel.withOption(ChannelOption.CONNECT_TIMEOUT_MILLIS, connectTimeoutMillis);
                channel.withOption(ChannelOption.SO_SNDBUF, 10 << 20); // 10 MiB
                channel.withOption(ChannelOption.SO_RCVBUF, 10 << 20); // 10 MiB
                channel.flowControlWindow(10 << 20); // 10 MiB
                channel.idleTimeout(30, TimeUnit.SECONDS);
                channel.keepAliveTime(1, TimeUnit.MINUTES);
                channel.keepAliveTimeout(30, TimeUnit.SECONDS);
                channel.keepAliveWithoutCalls(true);
            });
    }
}
