package ru.yandex.solomon.tool.alerting;

import java.io.BufferedReader;
import java.io.IOException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Optional;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Strings;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.alert.dao.codec.AlertCodec;
import ru.yandex.solomon.alert.dao.codec.AlertRecord;
import ru.yandex.solomon.alert.dao.ydb.YdbSchemaVersion;
import ru.yandex.solomon.alert.dao.ydb.entity.YdbAlertsDao;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;

import static ru.yandex.solomon.idempotency.IdempotentOperation.NO_OPERATION;

/**
 * @author Sergey Polovko
 */
public class RestoreFromBackup {

    public static void main(String[] args) throws IOException {
        if (args.length != 1) {
            System.err.println("Usage: tool <dump_file>");
            System.exit(1);
        }

        SolomonCluster cluster = SolomonCluster.PROD_FRONT;
        YdbClient ydbClient = YdbHelper.createYdbClient(cluster);
        BufferedReader reader = Files.newBufferedReader(Path.of(args[0]));

        ObjectMapper mapper = new ObjectMapper();
        AlertCodec codec = new AlertCodec(mapper);

        YdbAlertsDao dao = new YdbAlertsDao(
                cluster.kikimrRootPath(),
                ydbClient.table,
                ydbClient.scheme,
                YdbSchemaVersion.CURRENT,
                mapper);

        String line;
        while ((line = reader.readLine()) != null) {
            String[] cols = StringUtils.split(line, ',');

            AlertRecord rec = new AlertRecord();
            rec.projectId = parseString(cols[0]);
            rec.id = parseString(cols[1]);
            rec.version = Integer.parseInt(cols[2]);
            rec.name = parseString(cols[3]);
            rec.state = Integer.parseInt(cols[4]);
            rec.createdAt = Long.parseLong(cols[5]);
            rec.updatedAt = Long.parseLong(cols[6]);
            rec.groupByLabels = parseString(cols[7]);
            rec.notificationChannels = parseString(cols[8]);
            rec.type = Integer.parseInt(cols[9]);
            rec.config = parseString(cols[10]);
            rec.createdBy = parseString(cols[11]);
            rec.updatedBy = parseString(cols[12]);
            rec.annotations = parseString(cols[13]);
            rec.delaySeconds = Integer.parseInt(cols[14]);
            rec.description = parseString(cols[15]);
            rec.folderId = Strings.nullToEmpty(parseString(cols[16]));
            rec.notificationConfig = parseString(cols[17]);
            rec.noPointsPolicy = parseString(cols[18]);
            rec.resolvedEmptyPolicy = parseString(cols[19]);

            Alert alert = codec.decode(rec);

            Optional<Alert> saved = dao.update(alert, NO_OPERATION).join();
            System.out.printf("%5s %s\n", saved.isPresent() ? "ok" : "error", alert.getId());
        }

        System.exit(0);
    }

    static String parseString(String s) {
        int len = s.length();
        if (len > 0 && s.charAt(0) == '"' && s.charAt(len - 1) == '"') {
            // remove first and last quote char
            s = s.substring(1, len - 1);
        }

        if ("null".equals(s)) {
            return null;
        }

        return URLDecoder.decode(s, StandardCharsets.UTF_8);
    }
}
