package ru.yandex.solomon.tool.alerting.canonTests;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.zip.GZIPInputStream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.discovery.DiscoveryService;
import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.discovery.cluster.ClusterMapperImpl;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.metabase.client.MetabaseClientFactory;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.api.converters.AlertConverter;
import ru.yandex.solomon.alert.canon.Explainer;
import ru.yandex.solomon.alert.canon.Serialization;
import ru.yandex.solomon.alert.canon.protobuf.ArchiveConverterImpl;
import ru.yandex.solomon.alert.canon.protobuf.Converter;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertType;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.rule.ExplainResult;
import ru.yandex.solomon.alerting.canon.protobuf.TAlertEvaluationRecord;
import ru.yandex.solomon.config.SolomonConfigs;
import ru.yandex.solomon.config.protobuf.alert.TAlertingConfig;
import ru.yandex.solomon.config.thread.StubThreadPoolProvider;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.main.logger.LoggerConfigurationUtils;
import ru.yandex.solomon.metrics.client.MetricsClients;
import ru.yandex.solomon.metrics.client.RecordingMetricsClient;
import ru.yandex.solomon.util.PropertyInitializer;
import ru.yandex.stockpile.client.StockpileClientFactory;

import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class Recorder {

    static {
        PropertyInitializer.init();
    }

    private static final String EXTRACTED_ALERTS = ExtractAlertsToFiles.EXTRACTED_ALERTS;

    private static final String SENSORS_CLIENT_CAPTURE = "/tmp/capture.pb.gz";
    private static final String CANONICAL_EVALUATION_RESULT = "/tmp/evaluationResults.pb.gz";

    private static final Instant ALERT_EVALUATION_MOMENT = Instant.parse("2020-03-30T06:00:00Z");

    private static final String ALERTING_CONFIG = "/home/uranix/arcadia/solomon/configs/production/alerting.conf";

    public static void main(String[] args) throws IOException {
        // node clients are incredibly noisy
        LoggerConfigurationUtils.disableLogger();

        var totalAlerts = loadFrom(EXTRACTED_ALERTS);
        System.out.println("alerts loaded");
        var selectedAlerts = sample(totalAlerts);

        RecordingMetricsClient metricsClient = makeRecordingMetricsClient(ALERTING_CONFIG);

        System.out.println("Waiting for the clients to boot");
        while (true) {
            var mbAvail = metricsClient.getMetabaseAvailability().getAvailability();
            var spAvail = metricsClient.getStockpileAvailability().getAvailability();
            System.out.println("MB avail: " + mbAvail + ", SP avail: " + spAvail);

            if (mbAvail < 1 || spAvail < 1) {
                try {
                    Thread.sleep(2_000);
                } catch (InterruptedException ignore) {
                }
            } else {
                break;
            }
        }

        Explainer explainer = new Explainer(metricsClient);

        List<CompletableFuture<Pair<Alert, ExplainResult>>> taggedFutures = new ArrayList<>();
        for (var alertProto : selectedAlerts) {
            var alert = AlertConverter.protoToAlert(alertProto);
            taggedFutures.add(
                    explainer.unrollIfNecessary(alert)
                        .thenCompose(alertOrSubAlert -> {
                            if (alertOrSubAlert == null) {
                                return CompletableFuture.failedFuture(new IllegalStateException("empty multialert"));
                            }
                            return explainer.explain(alertOrSubAlert, ALERT_EVALUATION_MOMENT)
                                    .thenApply(er -> Pair.of(alertOrSubAlert, er))
                                    .whenComplete((alertKeyExplainResultPair, throwable) -> {
                                        if (throwable == null) {
                                            System.out.println(alertKeyExplainResultPair.getLeft().getKey());
                                        }
                                    });
                        })
                        .exceptionally(throwable -> Pair.of(null, null))
            );
        }

        var explainResults = join(CompletableFutures.allOf(taggedFutures))
                .filter(p -> p.getKey() != null);

        var countByStatus = explainResults.stream()
                .collect(Collectors.groupingBy(p -> p.getValue().getStatus().getCode(), Collectors.counting()));

        System.out.println("Stats: " + countByStatus);

        var capture = metricsClient.getCapture();

        System.out.println("Serializing capture to disk");
        try (OutputStream os = new FileOutputStream(SENSORS_CLIENT_CAPTURE)) {
            Serialization.writeMetricsClientCapture(capture, os);
        }

        System.out.println("Serializing done");

        System.out.println("Serializing evaluation to disk");

        List<TAlertEvaluationRecord> records = new ArrayList<>();
        for (var record : explainResults) {
            Alert alert = record.getKey();
            ExplainResult result = record.getValue();
            var builder = TAlertEvaluationRecord.newBuilder()
                    .setExplainResult(Converter.toProto(result));
            if (alert.getAlertType() == AlertType.SUB_ALERT) {
                builder.setSubAlert(AlertConverter.subAlertToProto((SubAlert) alert));
            } else {
                builder.setAlert(AlertConverter.alertToProto(alert));
            }
            records.add(builder.build());
        }

        try (OutputStream os = new FileOutputStream(CANONICAL_EVALUATION_RESULT)) {
            Serialization.writeCanonicalResults(ALERT_EVALUATION_MOMENT, records, os);
        }

        System.out.println("Serializing done");

        System.exit(0);
    }

    private static RecordingMetricsClient makeRecordingMetricsClient(String alertingConfig) {
        TAlertingConfig config = SolomonConfigs.parseConfig(alertingConfig, TAlertingConfig.getDefaultInstance());
        MetricRegistry registry = new MetricRegistry();
        ThreadPoolProvider threadPoolProvider = new StubThreadPoolProvider(8);
        var executor = threadPoolProvider.getExecutorService("CpuLowPriority", "");
        var timer = threadPoolProvider.getSchedulerExecutorService();
        ClusterMapper clusterMapper = new ClusterMapperImpl(config.getClustersConfigList(), DiscoveryService.async(), executor, timer);
        ClientOptionsFactory clientOptionsFactory = new ClientOptionsFactory(Optional.empty(), Optional.empty(), threadPoolProvider);
        MetabaseClientFactory mcf = new MetabaseClientFactory(threadPoolProvider, clusterMapper, registry, clientOptionsFactory);
        StockpileClientFactory scf = new StockpileClientFactory(threadPoolProvider, clusterMapper, registry, clientOptionsFactory);
        var metabaseClients = mcf.createClients("alert-extractor-recorder", config.getMetabaseClientConfig());
        var stockpileClients = scf.createClients("alert-extractor-recorder", config.getStockpileClientConfig());
        return RecordingMetricsClient.wrap(
                MetricsClients.create(metabaseClients, stockpileClients), ArchiveConverterImpl.I);
    }

    private static List<TAlert> sample(List<TAlert> alerts) {
        final int limit = 30;
        final long loadMillisLimit = Duration.ofHours(15).toMillis();

        return alerts.stream()
                .collect(Collectors.groupingBy(
                        a -> Pair.of(a.getProjectId(), a.getTypeCase()),
                        Collectors.collectingAndThen(
                                Collectors.toList(),
                                projectAlerts -> {
                                    Random random = new Random(42);
                                    Collections.shuffle(projectAlerts, random);
                                    ArrayList<TAlert> result = new ArrayList<>();
                                    long loadMillis = 0;
                                    int count = 0;
                                    for (var alert : projectAlerts) {
                                        loadMillis += alert.getPeriodMillis();
                                        count++;
                                        result.add(alert);
                                        if (loadMillis >= loadMillisLimit) {
                                            break;
                                        }
                                        if (count >= limit) {
                                            break;
                                        }
                                    }
                                    return result;
                                }
                        )
                ))
                .values()
                .stream()
                .flatMap(ArrayList::stream)
                .sorted(Comparator.comparing(TAlert::getProjectId).thenComparing(TAlert::getTypeCase))
                .collect(Collectors.toList());
    }

    private static List<TAlert> loadFrom(String path) throws IOException {
        List<TAlert> result = new ArrayList<>();

        InputStream is = new GZIPInputStream(new FileInputStream(path));
        while (true) {
            var alert = TAlert.parseDelimitedFrom(is);
            if (alert == null) {
                break;
            }
            result.add(alert);
        }

        return result;
    }
}
