package ru.yandex.solomon.tool.cfg;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import com.google.common.net.HostAndPort;

import ru.yandex.discovery.DiscoveryServices;


/**
 * @author Sergey Polovko
 */
public enum SolomonCluster {
    DEV_LOCAL("/Root", "/Root/Solomon/KV", "localhost"),
    DEV_MYT("/Root", "/Root/Solomon/KV", "solomon-dev-myt-00.search.yandex.net"),

    TEST_FRONT("/ru-prestable/solomon/development/solomon", "", "solomon-test-front-sas-00.search.yandex.net"),
    TEST("/Solomon", "/Solomon/Solomon/Stockpile/KV", "conductor_group://solomon_test_meta_storage"),

    PRESTABLE_FRONT("/ru-prestable/solomon/prestable/solomon", "", "conductor_group://solomon_pre_gateway"),
    PRESTABLE_FETCHER("/Solomon", "", "conductor_group://solomon_pre_fetcher"),
    PRESTABLE_STORAGE("/Solomon", "/Solomon/Solomon/Stockpile/KV", "conductor_group://solomon_pre_storage"),
    PRESTABLE_STOCKPILE("/Solomon", "/Solomon/Solomon/Stockpile/KV", "conductor_group://solomon_pre_stockpile"),

    PROD_FRONT("/Kfront/Solomon", "", "conductor_group://solomon_prod_gateway"),
    PROD_KFRONT("/Kfront/Solomon", "", "conductor_group://solomon_prod_kfront"),

    PROD_STORAGE_SAS("/Solomon", "/Solomon/Solomon/KV", "conductor_group://solomon_prod_storage_sas"),
    PROD_STORAGE_VLA("/Solomon", "/Solomon/Solomon/KV", "conductor_group://solomon_prod_storage_vla"),

    PROD_STOCKPILE_SAS("/Solomon", "/Solomon/Solomon/KV", "conductor_group://solomon_prod_stockpile_sas"),
    PROD_STOCKPILE_VLA("/Solomon", "/Solomon/Solomon/KV", "conductor_group://solomon_prod_stockpile_vla"),

    PROD_FETCHER_SAS("/Solomon", "", "conductor_group://solomon_prod_fetcher_sas"),
    PROD_FETCHER_VLA("/Solomon", "", "conductor_group://solomon_prod_fetcher_vla"),

    CLOUD_PROD_FRONT("/global/solomon", "", "conductor_group://cloud_prod_solomon-gateway"),

    CLOUD_PROD_STORAGE_VLA("/vla/solomon/meta", "/vla/solomon/Solomon", "conductor_group://cloud_prod_solomon-stockpile_vla"),
    CLOUD_PROD_STORAGE_SAS("/sas/solomon/meta", "/sas/solomon/Solomon", "conductor_group://cloud_prod_solomon-stockpile_sas"),

    CLOUD_PROD_FETCHER_VLA("/vla/solomon/meta", "", "conductor_group://cloud_prod_solomon-core_vla"),
    CLOUD_PROD_FETCHER_SAS("/sas/solomon/meta", "", "conductor_group://cloud_prod_solomon-core_sas"),

    CLOUD_PREPROD_FRONT("/pre-prod_global/solomon", "", "conductor_group://cloud_preprod_solomon-gateway"),

    ;

    private final String kikimrRootPath;
    private final String solomonVolumePath;
    private final List<String> addresses;

    SolomonCluster(String kikimrRootPath, String solomonVolumePath, String... hosts) {
        this.kikimrRootPath = kikimrRootPath;
        this.solomonVolumePath = solomonVolumePath;
        this.addresses = List.of(hosts);
    }

    public List<String> hosts() {
        return DiscoveryServices.resolve(addresses).stream().map(HostAndPort::getHost).collect(Collectors.toList());
    }

    public List<String> addresses() {
        return addresses;
    }

    public String kikimrRootPath() {
        return kikimrRootPath;
    }

    public String getSolomonVolumePath() {
        return solomonVolumePath;
    }

    public List<HostAndPort> addresses(int port) {
        return hosts().stream()
                .map(host -> HostAndPort.fromParts(host, port))
                .collect(Collectors.toList());
    }

    public List<HostAndPort> addressesKikimrGrpc() {
        return addresses(SolomonPorts.KIKIMR_GRPC);
    }

    public List<HostAndPort> addressesStockpileGrpc() {
        return addresses(SolomonPorts.STOCKPILE_GRPC);
    }

    // -- helpers --

    private static Stream<String> hostsSingle(String host) {
        return Stream.of(host);
    }

    private static Stream<String> hostsRange(String pattern, int from, int to) {
        return IntStream.rangeClosed(from, to)
            .mapToObj(i -> String.format(pattern, i));
    }
}
