package ru.yandex.solomon.tool.cleanup;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.OptionalLong;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.atomic.AtomicLong;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.yandex.ydb.table.query.Params;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.coremon.meta.db.MetricsDao;
import ru.yandex.solomon.coremon.meta.db.MetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.db.ydb.HugeTableSettings;
import ru.yandex.solomon.coremon.meta.db.ydb.YdbMetricsDaoFactory;
import ru.yandex.solomon.labels.intern.InterningLabelAllocator;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;

import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class DeleteLostMetricFromMetabase implements AutoCloseable {
    private final SolomonCluster cluster;
    private final YdbClient ydb;
    private final Path path;
    private final MetricsDaoFactory daoFactory;

    public DeleteLostMetricFromMetabase(SolomonCluster cluster, String file) {
        this.cluster = cluster;
        this.ydb = YdbHelper.createYdbClient(cluster);
        this.path = Path.of(file);
        this.daoFactory = YdbMetricsDaoFactory.forReadWrite(ydb.table, cluster.kikimrRootPath() + "/Solomon/Coremon/V1", MetricRegistry.root());
    }

    public static void main(String[] args) {
        if (args.length < 2) {
            System.err.println("Usage: tool <cluster_id> <file>");
            System.exit(1);
        }

        SolomonCluster cluster = SolomonCluster.valueOf(args[0]);
        String dir = args[1];
        try (var task = new DeleteLostMetricFromMetabase(cluster, dir)) {
            task.run();
        } catch (Throwable e) {
            e.printStackTrace();
            System.exit(1);
        }

        System.exit(0);
    }

    public void run() throws IOException {
        long totalShards = countShards();
        try (var reader = Files.newBufferedReader(path)) {
            AtomicLong shards = new AtomicLong();
            AtomicLong metrics = new AtomicLong();
            AsyncActorBody body = () -> {
                try {
                    var line = reader.readLine();
                    if (Strings.isNullOrEmpty(line)) {
                        return completedFuture(AsyncActorBody.DONE_MARKER);
                    }

                    var parts = StringUtils.split(line, ' ');
                    var shardId = parts[0];
                    var numId = Integer.parseUnsignedInt(parts[1]);
                    var count = Long.parseLong(parts[2]);

                    reportProgress(shards.get(), totalShards);
                    System.out.println("Processed shards " + DataSize.shortString(shards.get()));
                    System.out.println("Processed metrics " + DataSize.shortString(metrics.get()));
                    System.out.println("Deleting shard " + shardId + "...");
                    return deleteMetrics(numId, shardId, count)
                            .thenRun(() -> {
                                shards.incrementAndGet();
                                metrics.addAndGet(count);
                            });
                } catch (Throwable e) {
                    return failedFuture(e);
                }
            };

            var runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 10);
            runner.start().join();
            System.out.println("Done!");
        }
    }

    private void reportProgress(long count, long total) {
        double progress = count * 100. / total;
        System.out.println("Progress " + String.format("%.2f%%", progress));
    }

    public long countShards() {
        try (var stream = Files.lines(path)) {
            return stream.count();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public CompletableFuture<Void> deleteMetrics(int numId, String shardId, long count) {
        if (count == 0) {
            return completedFuture(null);
        } else if (count >= 200_000) {
            return deleteHugeShard(numId, shardId, count);
        }

        String table = cluster.kikimrRootPath() + "/Solomon/Coremon/V1/" + HugeTableSettings.makeName(numId);
        String query = String.format("""
                --!syntax_v1
                DECLARE $shardId as Uint32;
                DELETE FROM `%s` WHERE shardId = $shardId;
                """, table);
        Params params = Params.of("$shardId", uint32(numId));
        return ydb.fluent().retryForever().execute(query, params)
                .thenAccept(r -> r.expect("success delete metrics: " + Integer.toUnsignedLong(numId) + " shardId " + shardId));
    }

    public CompletableFuture<Void> deleteHugeShard(int numId, String shardId, long count) {
        var dao = daoFactory.create(numId, new InterningLabelAllocator());
        return deleteMetrics(dao, count);
    }

    public CompletableFuture<Void> deleteMetrics(MetricsDao dao, long count) {
        var config = RetryConfig.DEFAULT.withDelay(5_000)
                .withNumRetries(Integer.MAX_VALUE)
                .withExceptionFilter(throwable -> {
                    System.out.println("failed read metrics");
                    throwable.printStackTrace(System.out);
                    return true;
                });

        return RetryCompletableFuture.runWithRetries(() -> {
            return dao.findMetrics(array -> {
                var labels = new ArrayList<Labels>(array.size());
                for (int index = 0; index < array.size(); index++) {
                    labels.add(array.getLabels(index));
                }

                deleteMetrics(dao, labels).join();
            }, OptionalLong.of(count));
        }, config).thenApply(ignore -> null);
    }

    public CompletableFuture<Void> deleteMetrics(MetricsDao dao, List<Labels> labels) {
        CompletableFuture<Void> root = new CompletableFuture<>();
        CompletableFuture<Void> future = root;
        for (var part : Lists.partition(labels, 1000)) {
            future = future.thenCompose(ignore -> dao.deleteMetrics(part));
        }
        root.complete(null);
        return future;
    }

    @Override
    public void close() {
    }
}
