package ru.yandex.solomon.tool.cleanup;

import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.OptionalLong;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import com.google.common.net.HostAndPort;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.db.MetricsDao;
import ru.yandex.solomon.coremon.meta.db.MetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.db.ydb.YdbMetricsDaoFactory;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.tool.cfg.SolomonPorts;
import ru.yandex.solomon.tool.stockpile.Metric;
import ru.yandex.solomon.tool.stockpile.StockpileShardWriters;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.StockpileClientOptions;
import ru.yandex.stockpile.client.StockpileClients;
import ru.yandex.stockpile.client.StopStrategies;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class DeleteLostMetricFromStockpile implements AutoCloseable {

    private final SolomonCluster cluster;
    private final YdbClient ydb;
    private final MetricsDaoFactory daoFactory;
    private final StockpileClient stockpile;
    private final StockpileShardWriters stockpileWriter;
    private final StatsWriter statsWriter;

    public DeleteLostMetricFromStockpile(SolomonCluster cluster, String file) {
        this.cluster = cluster;
        this.ydb = YdbHelper.createYdbClient(cluster);
        this.daoFactory = YdbMetricsDaoFactory.forReadOnly(ydb.table, cluster.kikimrRootPath() + "/Solomon/Coremon/V1");
        this.stockpile = makeStockpileClient(cluster);
        this.stockpileWriter = new StockpileShardWriters(stockpile, ForkJoinPool.commonPool());
        this.statsWriter = new StatsWriter(Path.of(file));
    }

    private static StockpileClient makeStockpileClient(SolomonCluster cluster) {
        var opts = StockpileClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                        .setRpcExecutor(ForkJoinPool.commonPool())
                        .setRequestTimeOut(30, TimeUnit.SECONDS)
                        .setCircuitBreakerFailureQuantileThreshold(0.4))
                .setRetryStopStrategy(StopStrategies.alwaysStop())
                .setMetaDataRequestTimeOut(30, TimeUnit.SECONDS)
                .build();

        var addresses = cluster.hosts().stream()
                .map(host -> HostAndPort.fromParts(host, SolomonPorts.STOCKPILE_GRPC))
                .collect(Collectors.toList());

        System.out.println("addressed: " + addresses);
        return StockpileClients.create(addresses, opts);
    }

    public static void main(String[] args) {
        if (args.length < 2) {
            System.err.println("Usage: tool <cluster_id> <stats>");
            System.exit(1);
        }

        SolomonCluster cluster = SolomonCluster.valueOf(args[0]);
        String dir = args[1];
        try (var task = new DeleteLostMetricFromStockpile(cluster, dir)) {
            task.run();
        } catch (Throwable e) {
            e.printStackTrace();
            System.exit(1);
        }

        System.exit(0);
    }

    public void run() {
        var numIdToShardId = NumIdResolver.numIdToShardId(cluster);
        var numIdToShard = NumIdResolver.numIdToShard(cluster);

        List<Integer> toDelete = new ArrayList<>();
        for (var entry : numIdToShardId.int2ObjectEntrySet()) {
            if (!numIdToShard.containsKey(entry.getIntKey())) {
                toDelete.add(entry.getIntKey());
            }
        }
        Collections.shuffle(toDelete);

        var it = toDelete.iterator();
        AtomicLong metrics = new AtomicLong();
        AtomicLong shards = new AtomicLong();
        AtomicInteger processed = new AtomicInteger();
        AsyncActorBody body = () -> {
            if (!it.hasNext()) {
                return completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var numId = it.next();
            var shardId = numIdToShardId.get(numId);

            reportProgress(processed.incrementAndGet(), toDelete.size());
            System.out.println("Processed shards " + DataSize.shortString(shards.get()));
            System.out.println("Processed metrics " + DataSize.shortString(metrics.get()));

            return deleteMetrics(numId, shardId)
                    .thenAccept((count) -> {
                        shards.incrementAndGet();
                        metrics.addAndGet(count);
                        statsWriter.write(shardId, numId, count);
                    });
        };

        var runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 1);
        runner.start().join();
        System.out.println("Done!");
    }

    private void reportProgress(int count, long total) {
        double progress = count * 100. / total;
        System.out.println("Progress " + String.format("%.2f%%", progress));
    }

    public CompletableFuture<Long> deleteMetrics(int numId, String shardId) {
        try {
            var dao = daoFactory.create(numId);
            return countMetrics(dao)
                    .thenCompose(count -> {
                        if (count == 0) {
                            return completedFuture(0L);
                        }

                        return findMetrics(dao, count, this::deleteMetrics)
                                .thenApply(real -> Math.max(count, real));
                    });
        } catch (Throwable e) {
            return failedFuture(e);
        }
    }

    public void deleteMetrics(CoremonMetricArray chunk) {
        for (int i = 0; i < chunk.size(); i++) {
            int shardId = chunk.getShardId(i);
            long localId = chunk.getLocalId(i);

            var metric = new Metric(shardId, localId, DeleteMetricsFromStockpile.DELETE);
            stockpileWriter.write(metric).join();
        }
    }

    private CompletableFuture<Long> countMetrics(MetricsDao dao) {
        var config = RetryConfig.DEFAULT
                .withDelay(TimeUnit.SECONDS.toMillis(1))
                .withMaxDelay(TimeUnit.MINUTES.toMillis(5))
                .withNumRetries(Integer.MAX_VALUE)
                .withExceptionFilter(throwable -> {
                    System.out.println("failed count metrics");
                    throwable.printStackTrace(System.out);
                    return true;
                });

        return RetryCompletableFuture.runWithRetries(dao::getMetricCount, config);
    }

    private CompletableFuture<Long> findMetrics(MetricsDao dao, long expectCount, Consumer<CoremonMetricArray> consumer) {
        var config = RetryConfig.DEFAULT.withDelay(5_000)
                .withMaxDelay(TimeUnit.MINUTES.toMillis(5_000))
                .withNumRetries(Integer.MAX_VALUE)
                .withExceptionFilter(throwable -> {
                    System.out.println("failed read metrics");
                    throwable.printStackTrace(System.out);
                    return true;
                });
        return RetryCompletableFuture.runWithRetries(() -> dao.findMetrics(consumer, OptionalLong.of(expectCount)), config);
    }

    @Override
    public void close() {
        ydb.close();
        stockpileWriter.complete();
        stockpileWriter.doneFuture().join();
        stockpile.close();
    }

    private static class StatsWriter implements AutoCloseable {
        private final BufferedWriter writer;
        private final Queue<Stat> queue = new ConcurrentLinkedQueue<>();
        private final ActorRunner actor;
        private volatile boolean closed;

        public StatsWriter(Path path) {
            try {
                Files.createDirectories(path.getParent());
                writer = Files.newBufferedWriter(path, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
            this.actor = new ActorRunner(this::act, ForkJoinPool.commonPool());
        }

        private void act() {
            try {
                Stat next;
                while ((next = queue.poll()) != null) {
                    writer.write(next.shardId);
                    writer.write(" ");
                    writer.write(Integer.toUnsignedString(next.numId));
                    writer.write(" ");
                    writer.write(Long.toString(next.count));
                    writer.newLine();
                }

                if (closed) {
                    writer.close();
                }
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }

        public void write(String shardId, int numId, long count) {
            queue.add(new Stat(shardId, numId, count));
            actor.schedule();
        }

        @Override
        public void close() throws Exception {
            closed = true;
            actor.complete();
        }
    }

    private static record Stat(String shardId, int numId, long count) {
    }
}
