package ru.yandex.solomon.tool.cleanup;

import java.io.BufferedReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import com.yandex.ydb.table.TableClient;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.coremon.meta.db.MetricsDao;
import ru.yandex.solomon.coremon.meta.db.ydb.LabelListSortedSerialize;
import ru.yandex.solomon.coremon.meta.db.ydb.YdbMetricsDaoFactory;
import ru.yandex.solomon.labels.intern.InterningLabelAllocator;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;


/**
 * @author Sergey Polovko
 */
@SuppressWarnings("Duplicates")
public class DeleteMetricsFromMetabase {

    private static final int parallelism = 16;
    private static final ExecutorService threadPool = Executors.newFixedThreadPool(parallelism);
    private static final Semaphore processSemaphore = new Semaphore(parallelism);
    private static final AtomicLong deletedMetrics = new AtomicLong();

    public static void main(String[] args) {
        if (args.length < 3) {
            System.err.println("Usage: tool <cluster_id> <shard_id> <file>");
            System.exit(1);
        }

        SolomonCluster cluster = SolomonCluster.valueOf(args[0]);
        String shardId = args[1];
        String inputFile = args[2];
        int skipLines = (args.length == 4 ? Integer.parseInt(args[3]) : 0);
        int numId = NumIdResolver.resolveShardNumId(cluster, shardId);

        try (TableClient tableClient = YdbHelper.createTableClient(cluster)) {
            YdbMetricsDaoFactory metricFlatDao = YdbMetricsDaoFactory.forReadWrite(
                tableClient,
                cluster.kikimrRootPath() + "/Solomon/Coremon/V1",
                MetricRegistry.root());

            InterningLabelAllocator allocator = new InterningLabelAllocator();
            MetricsDao metricShardDao = metricFlatDao.create(numId, allocator);

            try (BufferedReader r = Files.newBufferedReader(Paths.get(inputFile))) {
                for (int i = 0; i < skipLines; i++) {
                    if (r.readLine() == null) break;
                }

                List<Labels> deleteBatch = new ArrayList<>(1000);
                String line;
                while ((line = r.readLine()) != null) {
                    deleteBatch.add(LabelListSortedSerialize.parse(line, allocator));

                    if (deleteBatch.size() >= 1000) {
                        doDelete(metricShardDao, deleteBatch);
                        deleteBatch = new ArrayList<>(1000);
                    }
                }

                if (!deleteBatch.isEmpty()) {
                    doDelete(metricShardDao, deleteBatch);
                }

                System.err.println("deletedMetrics: " + deletedMetrics);

                threadPool.shutdown();
                threadPool.awaitTermination(Long.MAX_VALUE, TimeUnit.MILLISECONDS);
            } catch (IOException e) {
                throw ExceptionUtils.throwException(e);
            }
        } catch (Exception e) {
            throw ExceptionUtils.throwException(e);
        }

        System.exit(0);
    }

    private static void doDelete(MetricsDao metricShardDao, List<Labels> keys) {
        try {
            processSemaphore.acquire();
        } catch (InterruptedException e) {
            throw ExceptionUtils.throwException(e);
        }

        threadPool.execute(() -> {
            metricShardDao.deleteMetrics(keys).join();
            System.err.println("deleted " + deletedMetrics.addAndGet(keys.size()) + " metrics");
            processSemaphore.release();
        });
    }
}
