package ru.yandex.solomon.tool.cleanup;

import java.io.BufferedReader;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiConsumer;

import it.unimi.dsi.fastutil.longs.LongArrayList;
import it.unimi.dsi.fastutil.longs.LongListIterator;

import ru.yandex.misc.io.ReaderSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.tool.StockpileHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.tool.stockpile.Metric;
import ru.yandex.solomon.tool.stockpile.StockpileShardWriters;
import ru.yandex.solomon.util.PropertyInitializer;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;


/**
 * @author Sergey Polovko
 */
public class DeleteMetricsFromStockpile {
    public static final MetricArchiveImmutable DELETE;
    static {
        var archive = new MetricArchiveMutable();
        archive.setDeleteBefore(DeleteBeforeField.DELETE_ALL);
        DELETE = archive.toImmutableNoCopy();
        PropertyInitializer.init();
    }

    public static void main(String[] args) {
        if (args.length < 2) {
            System.err.println("Usage: tool <cluster_id> <file_with_metric_ids> [--force]");
            System.exit(1);
        }

        SolomonCluster cluster = SolomonCluster.valueOf(args[0]);
        String filename = args[1];
        boolean force = args.length > 2 && "--force".equals(args[2]);

        try {
            new DeleteMetricsFromStockpile(cluster, force).run(filename);
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }

        System.exit(0);
    }

    private final StockpileClient stockpileClient;
    private final StockpileShardWriters writers;
    private final BiConsumer<Integer, LongArrayList> deleter;
    private final AtomicLong deletedMetrics = new AtomicLong();


    private DeleteMetricsFromStockpile(SolomonCluster cluster, boolean force) {
        this.stockpileClient = StockpileHelper.createGrpcClient(cluster);
        this.writers = new StockpileShardWriters(stockpileClient, ForkJoinPool.commonPool());
        this.deleter = force ? this::forceDeleter : this::fakeDeleter;
    }

    private void run(String filename) {
        stockpileClient.forceUpdateClusterMetaData().join();
        ReaderSource metricIdsSource = new File2(filename).asReaderTool();
        metricIdsSource.readBuffered(br -> {
            processLines(br, deleter);
            return null;
        });
        writers.complete();
        writers.doneFuture().join();
        stockpileClient.close();
    }

    private void forceDeleter(int shardId, LongArrayList localIds) {
        LongListIterator it = localIds.iterator();
        while (it.hasNext()) {
            var metric = new Metric(shardId, it.nextLong(), DELETE);
            writers.write(metric).join();
            var cnt = deletedMetrics.incrementAndGet();

            if (cnt % 10000 == 0) {
                System.err.println("deletedMetrics: " + deletedMetrics);
            }
        }
    }

    private void fakeDeleter(int shardId, LongArrayList localIds) {
        System.err.println("fake delete " + localIds.size() + " local ids from shard " + shardId);
    }

    private static void processLines(BufferedReader br, BiConsumer<Integer, LongArrayList> deleter) {
        int lineNo = 1;
        try {
            String line;
            int prevShardId = StockpileShardId.INVALID_SHARD_ID;
            LongArrayList localIds = new LongArrayList(100);

            while ((line = br.readLine()) != null) {
                line = firstColumn(line);
                String[] shardIdLocalIdStr = StringUtils.split(line, '/');
                int shardId = StockpileShardId.parse(shardIdLocalIdStr[0]);
                long localId = StockpileLocalId.parse(shardIdLocalIdStr[1]);

                if (shardId == prevShardId || prevShardId == StockpileShardId.INVALID_SHARD_ID) {
                    localIds.add(localId);
                } else {
                    deleter.accept(prevShardId, localIds);
                    localIds.clear();
                    localIds.add(localId);
                }

                prevShardId = shardId;
                lineNo++;
            }

            deleter.accept(prevShardId, localIds);
        } catch (Exception e) {
            throw new RuntimeException("error at line: " + lineNo, e);
        }
    }

    private static String firstColumn(String line) {
        int idx = line.indexOf(' ');
        if (idx == -1) {
            idx = line.indexOf('\t');
        }
        if (idx != -1) {
            line = line.substring(0, idx);
        }
        return line;
    }
}
