package ru.yandex.solomon.tool.cleanup;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ReadTableSettings;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;

import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.stockpile.client.shard.StockpileLocalId;

/**
 * @author Sergey Polovko
 */
public class DumpMetricIdsFromGraphite implements AutoCloseable {
    private final YdbClient ydb;
    private final Int2ObjectMap<BufferedWriter> writers;
    private final String tablePath;
    private final String dir;

    private DumpMetricIdsFromGraphite(String graphiteId, String dir) {
        SolomonCluster cluster = SolomonCluster.PROD_FRONT;
        this.ydb = YdbHelper.createYdbClient(cluster);
        this.writers = new Int2ObjectOpenHashMap<>();
        this.dir = dir;
        this.tablePath = cluster.kikimrRootPath() + "/MegaGraphite/" + graphiteId + "/Metrics";
    }

    private void run() {
        readTable(tablePath, r -> {
            try {
                int shardIdIdx = r.getColumnIndex("shardId");
                int localIdIdx = r.getColumnIndex("localId");
                while (r.next()) {
                    int shardId = r.getColumn(shardIdIdx).getInt32();
                    var writer = writer(shardId);
                    writer.write(StockpileLocalId.toString(r.getColumn(localIdIdx).getInt64() + '\n'));
                }
            } catch (IOException e) {
                throw new UncheckedIOException("cannot write id from " + tablePath, e);
            }
        });
    }

    private void readTable(String tablePath, Consumer<ResultSetReader> fn) {
        ydb.fluent()
                .executeOnSession(session -> {
                    var settings = ReadTableSettings.newBuilder()
                            .orderedRead(true)
                            .timeout(30, TimeUnit.MINUTES)
                            .build();

                    return session.readTable(tablePath, settings, fn);
                })
                .thenAccept(status -> status.expect("can not read table: " + tablePath))
                .join();
    }

    @Override
    public void close() throws Exception {
        for (var writer : writers.values()) {
            writer.close();
        }
        ydb.close();
    }

    private BufferedWriter writer(int shardId) {
        var result = writers.get(shardId);
        if (result != null) {
            return result;
        }

        try {
            Path filePath = Paths.get(dir).resolve(shardId + ".ids");
            result = Files.newBufferedWriter(filePath);
            writers.put(shardId, result);
            return result;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static void main(String[] args) {
        if (args.length != 2) {
            System.err.println("Usage: tool <graphite_id> <dir>");
            System.exit(1);
        }

        String graphiteId = args[0].toUpperCase();
        String dir = args[1];

        try (DumpMetricIdsFromGraphite tool = new DumpMetricIdsFromGraphite(graphiteId, dir)) {
            tool.run();
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }

        System.exit(0);
    }
}
