package ru.yandex.solomon.tool.cleanup;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import com.yandex.ydb.scheme.SchemeOperationProtos.Entry;
import com.yandex.ydb.scheme.SchemeOperationProtos.Entry.Type;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ReadTableSettings;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.ExceptionUtils;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.client.shard.StockpileLocalId;


/**
 * @author Sergey Polovko
 */
public class DumpMetricIdsFromMetabase implements AutoCloseable {

    private final int PARALLELISM = 16;
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withNumRetries(5)
            .withDelay(10_000)
            .withMaxDelay(60_000);

    private final YdbClient ydb;
    private final Int2ObjectMap<BufferedWriter> writers;
    private final String rootPath;
    private final String dir;
    private final ExecutorService executor;
    private final AtomicInteger dumpedTables = new AtomicInteger();
    private int totalTables = 0;

    private DumpMetricIdsFromMetabase(SolomonCluster cluster, String dir) {
        this.ydb = YdbHelper.createYdbClient(cluster);
        this.dir = dir;
        this.writers = new Int2ObjectOpenHashMap<>();
        this.rootPath = cluster.kikimrRootPath() + "/Solomon/metrics/";
        this.executor = Executors.newFixedThreadPool(PARALLELISM);
    }

    private void run() throws InterruptedException {
        var tables = ydb.scheme.listDirectory(rootPath)
                .join()
                .expect("can not list tablets").getChildren()
                .stream()
                .filter(entry -> entry.getType() == Type.TABLE)
                .map(Entry::getName)
                .collect(Collectors.toList());
        totalTables = tables.size();

        for (var table : tables) {
            executor.submit(() -> {
                try {
                    dumpIdsFromTable(table);
                } catch (Throwable t) {
                    Throwable cause = CompletableFutures.unwrapCompletionException(t);
                    ExceptionUtils.uncaughtException(t);
                }
            });
        }

        executor.shutdown();
        executor.awaitTermination(1, TimeUnit.DAYS);
    }

    private void dumpIdsFromTable(String tableName) {
        String tablePath = rootPath + tableName;

        System.err.printf(
            "[%05d/%05d] dumping %s (about %d rows)%n",
            dumpedTables.incrementAndGet(), totalTables, tableName, -1);

        readTable(tablePath, rs -> {
            try {
                int shardIdIdx = rs.getColumnIndex("shardId");
                int localIdIdx = rs.getColumnIndex("localId");
                while (rs.next()) {
                    int shardId = (int) rs.getColumn(shardIdIdx).getUint32();
                    BufferedWriter writer = writer(shardId);
                    writer.write(StockpileLocalId.toString(rs.getColumn(localIdIdx).getUint64()) + '\n');
                }
            } catch (IOException e) {
                throw new UncheckedIOException("cannot write id from " + tablePath, e);
            }
        });
    }

    private void readTable(String tablePath, Consumer<ResultSetReader> fn) {
        RetryCompletableFuture.runWithRetries(() -> {
            return ydb.fluent()
                    .executeOnSession(session -> {
                        var settings = ReadTableSettings.newBuilder()
                                .orderedRead(true)
                                .timeout(30, TimeUnit.MINUTES)
                                .build();

                        return session.readTable(tablePath, settings, fn);
                    })
                    .thenAccept(status -> status.expect("can not read table: " + tablePath));
        }, RETRY_CONFIG).join();
    }

    private BufferedWriter writer(int shardId) {
        var result = writers.get(shardId);
        if (result != null) {
            return result;
        }

        try {
            Path filePath = Paths.get(dir).resolve(shardId + ".ids");
            result = Files.newBufferedWriter(filePath);
            writers.put(shardId, result);
            return result;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void close() throws Exception {
        for (var writer : writers.values()) {
            writer.close();
        }
        ydb.close();
    }

    public static void main(String[] args) {
        if (args.length != 2) {
            System.err.println("Usage: tool <cluster_id> <dir>");
            System.exit(1);
        }

        SolomonCluster cluster = SolomonCluster.valueOf(args[0]);
        String dir = args[1];

        try (DumpMetricIdsFromMetabase tool = new DumpMetricIdsFromMetabase(cluster, dir)) {
            tool.run();
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }

        System.exit(0);
    }
}
