package ru.yandex.solomon.tool.cleanup;

import java.io.BufferedWriter;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Stream;

import com.google.protobuf.ByteString;

import ru.yandex.kikimr.client.KikimrAsyncRetry;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.codec.serializer.ByteStringsStockpile;
import ru.yandex.solomon.tool.KikimrHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.ExceptionUtils;
import ru.yandex.solomon.util.PropertyInitializer;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.kikimrKv.ShardIdMapToLong;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContentSerializer;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.IndexAddressWithFileCount;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.file.IndexFile;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * @author Stepan Koltsov
 * @author Sergey Polovko
 */
public class DumpMetricIdsFromStockpile {

    private static final int PARALLELISM = 8;

    private static KikimrKvClient kvClient;

    static {
        PropertyInitializer.init();
    }

    public static void main(String[] args) throws InterruptedException {
        if (args.length < 2) {
            System.err.println("Usage: tool <cluster_id> <dir> <fromShardId>");
            System.exit(1);
        }

        SolomonCluster cluster = SolomonCluster.valueOf(args[0]);
        final String dir = args[1];

        kvClient = KikimrHelper.createKvClient(cluster);
        var mapping = retry(() -> kvClient.resolveKvTablets(cluster.getSolomonVolumePath()))
            .thenApply(ShardIdMapToLong::new)
            .join();

        int shardIdFrom = args.length == 3 ? Integer.valueOf(args[2]) : 0;

        ExecutorService executor = Executors.newFixedThreadPool(PARALLELISM);

        mapping.shardIdStream()
            .filter(shardId -> shardId > shardIdFrom)
            .forEach(shardId -> {
                executor.submit(() -> {
                    try {
                        dumpMetricsIds(dir, mapping, shardId);
                    } catch (Throwable t) {
                        ExceptionUtils.uncaughtException(t);
                    }
                });
            });

        executor.shutdown();
        executor.awaitTermination(1, TimeUnit.DAYS);

        kvClient.close();
        System.exit(0);
    }

    private static void dumpMetricsIds(String dir, ShardIdMapToLong mapping, int shardId) {
        long tabletId = mapping.get(shardId);
        var localIds = retry(() -> listIndexes(tabletId)
            .thenCompose(files -> loadIdsFromIndex(tabletId, files)))
            .whenComplete((s, e) -> {
                if (e != null) {
                    System.err.println("failed dump metrics from shard " + shardId);
                    e.printStackTrace();
                    System.exit(1);
                }
            })
            .join();

        System.out.println(StockpileShardId.toString(shardId) + " (" + localIds.length + " metrics)");

        Path filePath = Paths.get(dir).resolve(shardId + ".ids");
        try (BufferedWriter writer = Files.newBufferedWriter(filePath)) {
            for (long localId : localIds) {
                writer.write(StockpileLocalId.toString(localId) + '\n');
            }
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }

    private static CompletableFuture<List<IndexAddressWithFileCount>> listIndexes(long tabletId) {
        return retry(() -> kvClient.readRangeNames(tabletId, 0, StockpileKvNames.currentFilesRange(), expiredAt()))
            .thenApply(response -> response.stream()
                .map(file -> FileNameParsed.parseCurrent(file.getName()))
                .filter(file -> file instanceof IndexFile)
                .map(IndexFile.class::cast)
                .collect(toList()))
            .thenApply(indices -> indices.stream()
                .collect(collectingAndThen(toList(), IndexAddressWithFileCount::fold)));
    }

    private static CompletableFuture<long[]> loadIdsFromIndex(long tabletId, List<IndexAddressWithFileCount> address) {
        IndexAddressWithFileCount indexName = address.stream()
            .filter(i -> i.getLevel() == SnapshotLevel.ETERNITY)
            .min(Comparator.comparingLong(IndexAddressWithFileCount::getTxn))
            .orElse(null);

        if (indexName == null) {
            return CompletableFuture.failedFuture(new IllegalStateException("not found snapshot at tablet " + tabletId + " only " + address));
        }

        return Stream.of(indexName.indexFileNames())
            .map(file -> retry(() -> kvClient.readData(tabletId, 0, file.reconstructCurrent(), expiredAt(), MsgbusKv.TKeyValueRequest.EPriority.BACKGROUND)))
            .collect(collectingAndThen(toList(), CompletableFutures::allOf))
            .thenApply(chunks -> {
                ByteString[] bytes = chunks.stream()
                    .map(buf -> ByteStringsStockpile.unsafeWrap(buf.get()))
                    .toArray(ByteString[]::new);

                SnapshotIndexContent indexContent = SnapshotIndexContentSerializer.S.deserializeParts(bytes);
                return indexContent.metricIdStream().toArray();
            });
    }

    private static <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return KikimrAsyncRetry.withRetries(KikimrAsyncRetry.RetryConfig.maxRetries(3), supplier);
    }

    private static long expiredAt() {
        return System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(5);
    }
}
