package ru.yandex.solomon.tool.cleanup;

import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.OptionalLong;

import com.yandex.ydb.table.TableClient;
import org.apache.commons.lang3.mutable.MutableLong;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.db.MetricsDao;
import ru.yandex.solomon.coremon.meta.db.MetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.db.ydb.Flags;
import ru.yandex.solomon.coremon.meta.db.ydb.LabelListSortedSerialize;
import ru.yandex.solomon.coremon.meta.db.ydb.YdbMetricsDaoFactory;
import ru.yandex.solomon.labels.intern.InterningLabelAllocator;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;


/**
 * @author Sergey Polovko
 */
public class DumpShardMeta {
    public static void main(String[] args) {
        if (args.length != 2) {
            System.err.println("Usage: tool <cluster_id> <shard_id>");
            System.exit(1);
        }

        SolomonCluster cluster = SolomonCluster.valueOf(args[0]);
        String shardId = args[1];
        int numId = NumIdResolver.resolveShardNumId(cluster, shardId);
        InterningLabelAllocator allocator = new InterningLabelAllocator();

        try (TableClient tableClient = YdbHelper.createTableClient(cluster)) {
            MetricsDaoFactory metricFlatDao = YdbMetricsDaoFactory.forReadOnly(
                tableClient,
                cluster.kikimrRootPath() + "/Solomon/Coremon/V1");
            MetricsDao metricShardDao = metricFlatDao.create(numId, allocator);
            MutableLong count = new MutableLong(0);

            try (BufferedWriter w = Files.newBufferedWriter(Paths.get(shardId + ".txt"))) {
                metricShardDao.findMetrics(chunk -> writeMeta(shardId, w, chunk, count), OptionalLong.empty()).join();
            } catch (IOException e) {
                throw ExceptionUtils.throwException(e);
            }
        } catch (Exception e) {
            throw ExceptionUtils.throwException(e);
        }
        System.exit(0);
    }

    public static void writeMeta(String progressPrefix, BufferedWriter w, CoremonMetricArray chunk, MutableLong count) {
        try {
            for (int i = 0; i < chunk.size(); i++) {
                w.write(StockpileShardId.toString(chunk.getShardId(i)));
                w.write('|');
                w.write(StockpileLocalId.toString(chunk.getLocalId(i)));
                w.write('|');
                w.write(LabelListSortedSerialize.format(chunk.getLabels(i)));
                w.write('|');
                w.write(Integer.toString(chunk.getCreatedAtSeconds(i)));
                w.write('|');
                w.write(Integer.toString(Flags.buildFlags(chunk.getType(i))));
                w.write('\n');

                long currentLine = count.incrementAndGet();
                if (currentLine % 1000 == 0) {
                    System.out.println(progressPrefix + " processed " + currentLine + " lines");
                }
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

    }

}
