package ru.yandex.solomon.tool.cleanup;

import java.util.Objects;
import java.util.concurrent.ForkJoinPool;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.yandex.ydb.core.auth.TokenAuthProvider;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;

import ru.yandex.solomon.core.db.dao.ydb.YdbShardsDao;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;

/**
 * @author Vladimir Gordiychuk
 */
public class NumIdResolver {

    public static int resolveShardNumId(SolomonCluster cluster, String shardId) {
        try (var ydb = configCluster(cluster)) {
            return ydb.client.fluent()
                    .execute("SELECT numId FROM [" + ydb.root + "/Config/V2/Shard] WHERE id = \"" + shardId + "\";")
                    .join()
                    .map(r -> {
                        var result = r.getResultSet(0);
                        Preconditions.checkState(result.next());
                        return result.getColumn("numId").getInt32();
                    })
                    .expect("success resolve numId");
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static Int2ObjectMap<String> numIdToShardId(SolomonCluster cluster) {
        try (var ydb = configCluster(cluster)) {
            YdbShardsDao dao = new YdbShardsDao(ydb.client.table, ydb.root + "/Config/V2/Shard", new ObjectMapper(), ForkJoinPool.commonPool());
            return dao.findAllIdToShardId().join();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static Int2ObjectMap<Shard> numIdToShard(SolomonCluster cluster) {
        try (var ydb = configCluster(cluster)) {
            YdbShardsDao dao = new YdbShardsDao(ydb.client.table, ydb.root + "/Config/V2/Shard", new ObjectMapper(), ForkJoinPool.commonPool());
            return dao.findAll()
                    .thenApply(shards -> {
                        Int2ObjectMap<Shard> result = new Int2ObjectOpenHashMap<>(shards.size());
                        for (var shard : shards) {
                            result.put(shard.getNumId(), shard);
                        }
                        return result;
                    })
                    .join();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static String getSelfSecret(String target) {
        return Objects.requireNonNull(System.getenv("YDB_TOKEN"), "please set YDB_TOKEN environment variable for " + target);
    }

    private static class Ydb implements AutoCloseable {
        private final String root;
        private final YdbClient client;

        public Ydb(String root, YdbClient client) {
            this.root = root;
            this.client = client;
        }

        public Ydb(SolomonCluster cluster) {
            root = cluster.kikimrRootPath();
            client = YdbHelper.createYdbClient(cluster);
        }

        @Override
        public void close() throws Exception {
            client.close();
        }
    }

    private static Ydb configCluster(SolomonCluster cluster) {
        switch (cluster){
            case PROD_STORAGE_VLA:
            case PROD_STORAGE_SAS:
            case PROD_FETCHER_SAS:
            case PROD_FETCHER_VLA:
            case PROD_KFRONT:
            case PROD_FRONT: {
                return new Ydb(SolomonCluster.PROD_KFRONT);
            }
            case PRESTABLE_FETCHER:
            case PRESTABLE_STORAGE:
            {
                var token = new TokenAuthProvider(getSelfSecret("ydb-ru-prestable.yandex.net:2135/ru-prestable/solomon/prestable/solomon"));
                var client = YdbHelper.createYdbClient(
                        "ydb-ru-prestable.yandex.net:2135",
                        "/ru-prestable/solomon/prestable/solomon",
                        token);
                var root = "/ru-prestable/solomon/prestable/solomon";
                return new Ydb(root, client);
            }
            case TEST:
            {
                var token = new TokenAuthProvider(getSelfSecret("ydb-ru-prestable.yandex.net:2135/ru-prestable/solomon/development/solomon"));
                var client = YdbHelper.createYdbClient(
                        "ydb-ru-prestable.yandex.net:2135",
                        "/ru-prestable/solomon/development/solomon",
                        token);
                var root = "/ru-prestable/solomon/development/solomon";
                return new Ydb(root, client);
            }
            case CLOUD_PREPROD_FRONT:
                return new Ydb(SolomonCluster.CLOUD_PREPROD_FRONT);
            case CLOUD_PROD_FETCHER_SAS:
            case CLOUD_PROD_FETCHER_VLA:
            case CLOUD_PROD_STORAGE_SAS:
            case CLOUD_PROD_STORAGE_VLA:
            case CLOUD_PROD_FRONT:
                return new Ydb(SolomonCluster.CLOUD_PROD_FRONT);
            default:
                throw new UnsupportedOperationException("Unknown cluster: " + cluster);
        }
    }
}
