package ru.yandex.solomon.tool.cleanup;

import java.io.BufferedReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import com.yandex.ydb.table.TableClient;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.coremon.meta.CoremonMetricArray;
import ru.yandex.solomon.coremon.meta.db.MetricsDao;
import ru.yandex.solomon.coremon.meta.db.MetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.db.ydb.Flags;
import ru.yandex.solomon.coremon.meta.db.ydb.LabelListSortedSerialize;
import ru.yandex.solomon.coremon.meta.db.ydb.YdbMetricsDaoFactory;
import ru.yandex.solomon.labels.intern.InterningLabelAllocator;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;


/**
 * @author Sergey Polovko
 */
@SuppressWarnings("Duplicates")
public class RestoreMetricsInMetabase {

    private static final int parallelism = 16;
    private static final ExecutorService threadPool = Executors.newFixedThreadPool(parallelism);
    private static final Semaphore processSemaphore = new Semaphore(parallelism);
    private static final AtomicLong insertedMetrics = new AtomicLong();

    public static void main(String[] args) {
        if (args.length < 3) {
            System.err.println("Usage: tool <cluster_id> <shard_id> <file>");
            System.exit(1);
        }

        SolomonCluster cluster = SolomonCluster.valueOf(args[0]);
        String shardId = args[1];
        String inputFile = args[2];
        int skipLines = (args.length == 4 ? Integer.parseInt(args[3]) : 0);
        int numId = NumIdResolver.resolveShardNumId(cluster, shardId);

        try (TableClient tableClient = YdbHelper.createTableClient(cluster)) {
            MetricsDaoFactory metricFlatDao = YdbMetricsDaoFactory.forReadWrite(
                tableClient,
                cluster.kikimrRootPath() + "/Solomon/Coremon/V1",
                MetricRegistry.root());

            InterningLabelAllocator allocator = new InterningLabelAllocator();
            MetricsDao metricShardDao = metricFlatDao.create(numId, allocator);

            try (BufferedReader r = Files.newBufferedReader(Paths.get(inputFile))) {
                for (int i = 0; i < skipLines; i++) {
                    if (r.readLine() == null) break;
                }

                try (CoremonMetricArray insertBatch = new CoremonMetricArray(1000)) {
                    String line;
                    while ((line = r.readLine()) != null) {
                        String[] columns = StringUtils.split(line, '|');
                        int stockpileShardId = StockpileShardId.parse(columns[0]);
                        long stockpileLocalId = StockpileLocalId.parse(columns[1]);
                        Labels parse = LabelListSortedSerialize.parse(columns[2], allocator);
                        int createdAtSeconds = Integer.parseInt(columns[3]);
                        MetricType metricType = Flags.readMetricType(Integer.parseInt(columns[4]));

                        insertBatch.add(stockpileShardId, stockpileLocalId, parse, createdAtSeconds, metricType);

                        if (insertBatch.size() >= 1000) {
                            doInsert(metricShardDao, insertBatch);
                            insertBatch.clear();
                        }
                    }

                    if (!insertBatch.isEmpty()) {
                        doInsert(metricShardDao, insertBatch);
                    }
                }

                System.err.println("insertedMetrics: " + insertedMetrics);

                threadPool.shutdown();
                threadPool.awaitTermination(Long.MAX_VALUE, TimeUnit.MILLISECONDS);
            } catch (IOException e) {
                throw ExceptionUtils.throwException(e);
            }
        } catch (Exception e) {
            throw ExceptionUtils.throwException(e);
        }

        System.exit(0);
    }

    private static void doInsert(MetricsDao metricShardDao, CoremonMetricArray metrics) {
        try {
            processSemaphore.acquire();
        } catch (InterruptedException e) {
            throw ExceptionUtils.throwException(e);
        }

        threadPool.execute(() -> {
            metricShardDao.replaceMetrics(metrics).join();
            System.err.println("inserted " + insertedMetrics.addAndGet(metrics.size()) + " metrics");
            processSemaphore.release();
        });
    }
}
