package ru.yandex.solomon.tool.mg;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;

import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.transaction.TxControl;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructType;
import com.yandex.ydb.table.values.Value;

import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;

/**
 * @author Sergey Polovko
 */
public class DeleteMetrics {

    private static final int MAX_BATCH_SIZE = 10_000;

    private static final StructType KEY_TYPE = StructType.of(
        "hash", PrimitiveType.uint32(),
        "name", PrimitiveType.utf8());

    private static final ListType KEYS_LIST_TYPE = ListType.of(KEY_TYPE);


    public static void main(String[] args) {
        if (args.length != 2) {
            System.err.println("Usage: tool {ps|gr|bs} <input_file>");
            System.exit(1);
        }

        String graphiteId = args[0].toUpperCase();
        Path inputFile = Path.of(args[1]);

        String deleteQuery = String.format("""
                --!syntax_v1
                DECLARE $keys AS %s;
                DELETE FROM `/Kfront/MegaGraphite/%s/Metrics` ON SELECT * FROM AS_TABLE($keys);
                """, KEYS_LIST_TYPE, graphiteId);

        try (
            TableClient tableClient = YdbHelper.createTableClient(SolomonCluster.PROD_FRONT);
            BufferedReader reader = Files.newBufferedReader(inputFile))
        {
            var retryCtx = SessionRetryContext.create(tableClient)
                .maxRetries(10)
                .backoffSlot(Duration.ofSeconds(1))
                .sessionSupplyTimeout(Duration.ofSeconds(10))
                .build();

            int deleted = 0;

            while (true) {
                List<Value> batch = readNextBatch(reader);
                if (batch.isEmpty()) {
                    break;
                }

                Params params = Params.of("$keys", KEYS_LIST_TYPE.newValue(batch));
                var result = retryCtx.supplyResult(s -> s.executeDataQuery(deleteQuery, TxControl.serializableRw(), params)).join();
                result.expect("cannot delete metrics");

                deleted += batch.size();
                System.err.printf("\u001b[1000Ddeleted %9d rows", deleted);
            }
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }

        System.err.println("\ndone");
        System.exit(0);
    }

    private static List<Value> readNextBatch(BufferedReader reader) {
        var batch = new ArrayList<Value>(MAX_BATCH_SIZE);

        try {
            for (int i = 0; i < MAX_BATCH_SIZE; i++) {
                String line = reader.readLine();
                if (line == null) {
                    break;
                }

                // NOTE: unsafe construction, do not modify order of parameters!
                batch.add(KEY_TYPE.newValueUnsafe(
                    /* hash */ PrimitiveValue.uint32(line.hashCode()),
                    /* name */ PrimitiveValue.utf8(line)));
            }
        } catch (IOException e) {
            throw new UncheckedIOException("cannot read next line from input file", e);
        }

        return batch;
    }
}
