package ru.yandex.solomon.tool.mg;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.io.Writer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import com.yandex.ydb.core.Status;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.settings.ReadTableSettings;

import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.stockpile.client.shard.StockpileMetricId;

import static java.nio.file.StandardOpenOption.CREATE;
import static java.nio.file.StandardOpenOption.TRUNCATE_EXISTING;
import static java.nio.file.StandardOpenOption.WRITE;

/**
 * @author Sergey Polovko
 */
public class DumpMetrics {

    private static CompletableFuture<Status> readTable(Session session, String tablePath, Writer writer) {
        ReadTableSettings settings = ReadTableSettings.newBuilder()
            .timeout(1, TimeUnit.HOURS)
            .columns("name", "shardId", "localId")
            .orderedRead(false)
            .build();

        AtomicInteger rows = new AtomicInteger(0);
        return session.readTable(tablePath, settings, resultSet -> {
            int nameIdx = resultSet.getColumnIndex("name");
            int shardIdIdx = resultSet.getColumnIndex("shardId");
            int localIdIdx = resultSet.getColumnIndex("localId");

            while (resultSet.next()) {
                String name = resultSet.getColumn(nameIdx).getUtf8();
                int shardId = resultSet.getColumn(shardIdIdx).getInt32();
                long localId = resultSet.getColumn(localIdIdx).getInt64();

                try {
                    writer.write(StockpileMetricId.toString(shardId, localId));
                    writer.write(' ');
                    writer.write(name);
                    writer.write('\n');
                } catch (IOException e) {
                    throw new UncheckedIOException("cannot write metric " + name, e);
                }
            }

            int written = rows.addAndGet(resultSet.getRowCount());
            System.err.printf("\u001b[1000Dwritten %9d rows", written);
        });
    }

    public static void main(String[] args) {
        if (args.length != 2) {
            System.err.println("Usage: tool {ps|gr|bs} <output_file>");
            System.exit(1);
        }

        String graphiteId = args[0].toUpperCase();
        Path outputFile = Path.of(args[1]);

        try (
            TableClient tableClient = YdbHelper.createTableClient(SolomonCluster.PROD_FRONT);
            Writer writer = Files.newBufferedWriter(outputFile, CREATE, TRUNCATE_EXISTING, WRITE))
        {
            var retryCtx = SessionRetryContext.create(tableClient)
                .maxRetries(10)
                .backoffSlot(Duration.ofSeconds(1))
                .sessionSupplyTimeout(Duration.ofSeconds(10))
                .build();

            String tablePath = "/Kfront/MegaGraphite/" + graphiteId + "/Metrics";
            var status = retryCtx.supplyStatus(session -> readTable(session, tablePath, writer)).join();
            status.expect("cannot read table");
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }

        System.err.println("\ndone");
        System.exit(0);
    }
}
