package ru.yandex.solomon.tool.mg;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.io.Writer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.longs.LongSet;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.solomon.tool.StockpileHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.MetricMeta;
import ru.yandex.stockpile.api.ReadMetricsMetaRequest;
import ru.yandex.stockpile.api.ReadMetricsMetaResponse;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileMetricId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static java.nio.file.StandardOpenOption.CREATE;
import static java.nio.file.StandardOpenOption.TRUNCATE_EXISTING;
import static java.nio.file.StandardOpenOption.WRITE;


/**
 * @author Sergey Polovko
 */
public class FilterStaleMetrics {
    private static final RetryConfig RETRY = RetryConfig.DEFAULT
            .withDelay(TimeUnit.SECONDS.toMillis(1))
            .withMaxDelay(TimeUnit.MINUTES.toMillis(5))
            .withNumRetries(Integer.MAX_VALUE)
            .withExceptionFilter(throwable -> {
                System.out.println("failed load meta");
                throwable.printStackTrace(System.out);
                return true;
            });

    private static final int PARALLELISM = 16;
    private static final int MAX_BATCH_SIZE = 10_000;

    interface MetricsConsumer {
        void accept(int shardId, Long2ObjectOpenHashMap<String> localId2Name);
    }

    private static void readMetrics(Path inputFile, MetricsConsumer consumer) {
        try (BufferedReader reader = Files.newBufferedReader(inputFile)) {
            String line;
            int prevShardId = StockpileShardId.INVALID_SHARD_ID;
            Long2ObjectOpenHashMap<String> localId2Name = new Long2ObjectOpenHashMap<>(1000);

            while ((line = reader.readLine()) != null) {
                int delimIndex = line.indexOf(' ');
                String metricIdStr = line.substring(0, delimIndex);
                String name = line.substring(delimIndex + 1);

                int i = metricIdStr.indexOf('/');
                int shardId = StockpileShardId.parse(metricIdStr.substring(0, i));
                long localId = StockpileLocalId.parse(metricIdStr.substring(i + 1));

                if (shardId == prevShardId || prevShardId == StockpileShardId.INVALID_SHARD_ID) {
                    localId2Name.put(localId, name);
                    if (localId2Name.size() >= MAX_BATCH_SIZE) {
                        consumer.accept(prevShardId, localId2Name);
                        localId2Name = new Long2ObjectOpenHashMap<>(1000);
                    }
                } else {
                    consumer.accept(prevShardId, localId2Name);
                    localId2Name = new Long2ObjectOpenHashMap<>(1000);
                    localId2Name.put(localId, name);
                }

                prevShardId = shardId;
            }
        } catch (IOException e) {
            throw ExceptionUtils.throwException(e);
        }
    }

    private static List<MetricMeta> loadMeta(StockpileClient client, int shardId, LongSet localIds) {
        var request = ReadMetricsMetaRequest.newBuilder()
            .setShardId(shardId)
            .addAllLocalIds(localIds)
            .build();

        ReadMetricsMetaResponse response = readMeta(client, request).join();
        return response.getMetaList();
    }

    private static CompletableFuture<ReadMetricsMetaResponse> readMeta(StockpileClient client, ReadMetricsMetaRequest request) {
        return RetryCompletableFuture.runWithRetries(() -> client.readMetricsMeta(request)
                .thenApply(response -> {
                    if (EStockpileStatusCode.OK != response.getStatus()) {
                        throw new StockpileRuntimeException(response.getStatus(), response.getStatusMessage());
                    }
                    return response;
                }), RETRY);
    }

    private static int writeMetrics(Writer writer, long removeAfterMillis, Chunk chunk) {
        try {
            int toDelete = 0;
            for (MetricMeta m : chunk.meta) {
                if (m.getLastTsMillis() != 0 && m.getLastTsMillis() < removeAfterMillis) {
                    toDelete++;
                    writer.write(InstantUtils.formatToSeconds(m.getLastTsMillis()));
                    writer.write(' ');
                    writer.write(StockpileMetricId.toString(chunk.shardId, m.getLocalId()));
                    writer.write(' ');
                    writer.write(chunk.localId2Name.get(m.getLocalId()));
                    writer.write('\n');
                }
            }
            return toDelete;
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static void main(String[] args) {
        if (args.length != 3) {
            System.err.println("Usage: tool <input_file> <output_file> <remove_after_days>");
            System.exit(1);
        }

        Path inputFile = Path.of(args[0]);
        Path outputFile = Path.of(args[1]);
        int removeAfterDays = Integer.parseInt(args[2]);

        try (
            Writer writer = Files.newBufferedWriter(outputFile, CREATE, TRUNCATE_EXISTING, WRITE);
            StockpileClient stockpileClient = StockpileHelper.createGrpcClient(SolomonCluster.PROD_STOCKPILE_VLA))
        {
            stockpileClient.forceUpdateClusterMetaData().join();

            var inputQueue = new ArrayBlockingQueue<Chunk>(100);
            var outputQueue = new ArrayBlockingQueue<Chunk>(100);

            ExecutorService threadPool = Executors.newFixedThreadPool(PARALLELISM + 1);

            AtomicBoolean readerRunning = new AtomicBoolean(true);
            threadPool.submit(() -> {
                readMetrics(inputFile, (shardId, localId2Name) -> {
                    try {
                        inputQueue.put(new Chunk(shardId, localId2Name));
                    } catch (InterruptedException e) {
                        ru.yandex.solomon.util.ExceptionUtils.uncaughtException(e);
                    }
                });
                readerRunning.set(false);
            });

            AtomicInteger workersRunning = new AtomicInteger(PARALLELISM);
            for (int i = 0; i < PARALLELISM; i++) {
                threadPool.submit(() -> {
                    try {
                        while (true) {
                            Chunk chunk = readerRunning.get() ? inputQueue.take() : inputQueue.poll();
                            if (chunk == null) {
                                break;
                            }
                            chunk.meta = loadMeta(stockpileClient, chunk.shardId, chunk.localId2Name.keySet());
                            outputQueue.put(chunk);
                        }

                        workersRunning.decrementAndGet();
                    } catch (InterruptedException e) {
                        ru.yandex.solomon.util.ExceptionUtils.uncaughtException(e);
                    }
                });
            }

            int rowsTotal = 0, rowsDelete = 0;
            long removeAfterMillis = Instant.now().minus(Duration.ofDays(removeAfterDays)).toEpochMilli();

            while (true) {
                Chunk chunk = workersRunning.get() == 0 ? outputQueue.poll() : outputQueue.take();
                if (chunk == null) {
                    break;
                }

                rowsDelete += writeMetrics(writer, removeAfterMillis, chunk);

                rowsTotal += chunk.localId2Name.size();
                System.err.printf("\u001b[1000Dprocessed %9d rows", rowsTotal);
            }

            System.err.printf("gc: %d -> %d (%d metrics or %.1f%%)\n",
                rowsTotal,
                rowsTotal - rowsDelete,
                rowsDelete,
                (rowsDelete * 100. / (double) rowsTotal));

            threadPool.shutdown();
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }

        System.exit(0);
    }

    private static final class Chunk {
        final int shardId;
        final Long2ObjectOpenHashMap<String> localId2Name;
        List<MetricMeta> meta;

        public Chunk(int shardId, Long2ObjectOpenHashMap<String> localId2Name) {
            this.shardId = shardId;
            this.localId2Name = localId2Name;
        }
    }
}
