package ru.yandex.solomon.tool.mg;

import java.util.Set;
import java.util.concurrent.TimeUnit;

import com.yandex.ydb.table.query.Params;

import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.metrics.client.TimeSeriesCodec;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.tool.StockpileHelper;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.collection.array.LongArrayView;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.client.shard.StockpileMetricId;

import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;


/**
 * @author Sergey Polovko
 */
public class ReadMetric {

    private static final Set<String> KNOWN_MG = Set.of("PS", "GR", "BS");

    private static StockpileMetricId getStockpileId(String mg, String metric) {
        String query = String.format("""
                --!syntax_v1
                DECLARE $hash as Uint32;
                DECLARE $name as Utf8;

                SELECT shardId, localId
                FROM `/Kfront/MegaGraphite/%s/Metrics`
                WHERE hash = $hash AND name = $name;
                """, mg);

        var params = Params.of("$hash", uint32(metric.hashCode()), "$name", utf8(metric));

        try (var ydb = YdbHelper.createYdbClient(SolomonCluster.PROD_FRONT)) {
            return ydb.fluent().execute(query, params)
                    .thenApply(result -> {
                        var rs = result.expect("success").getResultSet(0);
                        if (!rs.next()) {
                            throw new IllegalStateException("metric not found: " + metric);
                        }

                        int shardId = rs.getColumn("shardId").getInt32();
                        long localId = rs.getColumn("localId").getInt64();
                        return new StockpileMetricId(shardId, localId);
                    }).join();
        }
    }

    public static void main(String[] args) {
        if (args.length < 2) {
            System.err.println("Usage: tool {PS|BS|GR} <metric>");
            System.exit(1);
        }

        String mg = args[0];
        String metric = args[1];
        if (!KNOWN_MG.contains(mg)) {
            System.err.println("Unknown MegaGraphite instance: " + mg);
            System.err.println("Known instances are " + KNOWN_MG);
            System.exit(1);
        }

        try {
            StockpileMetricId id = getStockpileId(mg, metric);
            System.out.println(metric + " => " + id);

            GraphData sasData = loadData(SolomonCluster.PROD_STOCKPILE_SAS, id);
            GraphData vlaData = loadData(SolomonCluster.PROD_STOCKPILE_VLA, id);

            int sasIdx = 0, sasLen = sasData.length();
            LongArrayView sasTime = sasData.getTimestamps();

            int vlaIdx = 0, vlaLen = vlaData.length();
            LongArrayView vlaTime = vlaData.getTimestamps();

            while (sasIdx < sasLen && vlaIdx < vlaLen) {
                long sasTs = sasTime.at(sasIdx);
                long vlaTs = vlaTime.at(vlaIdx);

                if (sasTs == vlaTs) {
                    double sasValue = sasData.getValues().at(sasIdx++);
                    double vlaValue = vlaData.getValues().at(vlaIdx++);
                    System.out.println(InstantUtils.formatToSeconds(sasTs) + " => (sas: " + sasValue + ", vla: " + vlaValue + ")");
                } else if (sasTs < vlaTs) {
                    double sasValue = sasData.getValues().at(sasIdx++);
                    System.out.println(InstantUtils.formatToSeconds(sasTs) + " => (sas: " + sasValue + ")");
                } else {
                    double vlaValue = vlaData.getValues().at(vlaIdx++);
                    System.out.println(InstantUtils.formatToSeconds(vlaTs) + " => (vla: " + vlaValue + ")");
                }
            }

            for (; sasIdx < sasLen; sasIdx++) {
                long sasTs = sasTime.at(sasIdx);
                double sasValue = sasData.getValues().at(sasIdx);
                System.out.println(InstantUtils.formatToSeconds(sasTs) + " => (sas: " + sasValue + ")");
            }

            for (; vlaIdx < vlaLen; vlaIdx++) {
                long vlaTs = vlaTime.at(vlaIdx);
                double vlaValue = vlaData.getValues().at(vlaIdx);
                System.out.println(InstantUtils.formatToSeconds(vlaTs) + " => (vla: " + vlaValue + ")");
            }
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }

        System.exit(0);
    }

    private static GraphData loadData(SolomonCluster cluster, StockpileMetricId id) {
        try (var client = StockpileHelper.createGrpcClient(cluster)) {
            client.forceUpdateClusterMetaData().join();

            var response = client.readCompressedOne(TReadRequest.newBuilder()
                .setMetricId(MetricId.newBuilder()
                    .setShardId(id.getShardId())
                    .setLocalId(id.getLocalId())
                    .build())
                .setBinaryVersion(StockpileFormat.CURRENT.getFormat())
                .setDeadline(System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(1))
                .build())
                .join();

            if (response.getStatus() != EStockpileStatusCode.OK) {
                throw new StockpileRuntimeException(response.getStatus(), response.getStatusMessage());
            }

            return AggrGraphDataArrayList.of(TimeSeriesCodec.sequenceDecode(response)).toGraphDataShort();
        }
    }
}
