package ru.yandex.solomon.tool.migration.kv;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;

import ru.yandex.monlib.metrics.primitives.Counter;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.LazyGaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * @author Vladimir Gordiychuk
 */
public class MetabaseTransferMetrics {
    private final MetricRegistry registry;
    private final ConcurrentMap<String, ShardMetrics> metrics = new ConcurrentHashMap<>();

    final GaugeInt64 shardsCount;
    final Counter shardsProcessed;
    final LazyGaugeInt64 shardsWaiting;
    final Counter retryCount;


    public MetabaseTransferMetrics(MetricRegistry registry) {
        this.registry = registry;
        this.shardsCount = registry.gaugeInt64("metabase.shard.count");
        this.shardsProcessed = registry.counter("metabase.shard.processed");
        this.shardsWaiting = registry.lazyGaugeInt64("metabase.shard.waiting", () -> shardsCount.get() - shardsProcessed.get());
        this.retryCount = registry.counter("metabase.migration.retry.count");
    }

    public ShardMetrics getShardMetrics(String shardId) {
        return metrics.computeIfAbsent(shardId, id -> new ShardMetrics(id, registry));
    }

    public static class ShardMetrics {
        final GaugeInt64 metricsTotal;
        final Counter metricsLoad;
        final Counter metricsWrite;

        volatile long startedAtNanos;
        volatile long completedAtNanos;

        public ShardMetrics(String shardId, MetricRegistry registry) {
            this(registry.subRegistry("shardId", shardId));
        }

        public ShardMetrics(MetricRegistry registry) {
            metricsTotal = registry.gaugeInt64("metabase.shard.metrics.total");
            metricsLoad = registry.counter("metabase.shard.metrics.load");
            metricsWrite = registry.counter("metabase.shard.metrics.write");

            registry.lazyCounter("metabase.shard.waiting.metrics.load",
                    () -> metricsTotal.get() - metricsLoad.get());

            registry.lazyCounter("metabase.shard.waiting.metrics.write",
                    () -> metricsTotal.get() - metricsWrite.get());

            registry.lazyCounter("metabase.shard.migration.timeMs", () -> {
                if (startedAtNanos == 0) {
                    return 0;
                }

                if (completedAtNanos == 0) {
                    return TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startedAtNanos);
                }

                return TimeUnit.NANOSECONDS.toMillis(completedAtNanos - startedAtNanos);
            });
        }
    }
}
