package ru.yandex.solomon.tool.migration.kv;

import java.io.ByteArrayOutputStream;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import io.netty.handler.codec.http.HttpStatusClass;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.util.HttpConstants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.thread.factory.DaemonThreadFactory;
import ru.yandex.misc.thread.factory.ThreadNameThreadFactory;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.MetricEncoder;
import ru.yandex.monlib.metrics.encode.MetricFormat;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.histogram.HistogramSnapshot;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.util.host.HostUtils;

import static io.netty.handler.codec.http.HttpHeaderNames.AUTHORIZATION;
import static io.netty.handler.codec.http.HttpHeaderNames.CONTENT_TYPE;
import static ru.yandex.monlib.metrics.encode.MetricEncoderFactory.createEncoder;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricsPushClient implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(MetricsPushClient.class);
    private final AsyncHttpClient httpClient;
    private final ScheduledExecutorService executorService;

    private MetricsPushClient(AsyncHttpClient httpClient, ScheduledExecutorService executorService) {
        this.httpClient = httpClient;
        this.executorService = executorService;
    }

    public static MetricsPushClient create() {
        AsyncHttpClient httpClient = new DefaultAsyncHttpClient();
        ScheduledExecutorService executorService =
                Executors.newSingleThreadScheduledExecutor(new DaemonThreadFactory(new ThreadNameThreadFactory(
                    MetricsPushClient.class)));

        return new MetricsPushClient(httpClient, executorService);
    }

    public void schedulePush(ShardKey shard, MetricSupplier supplier) {
        schedulePush(shard, supplier, 15, TimeUnit.SECONDS);
    }

    public void schedulePush(ShardKey shard, MetricSupplier supplier, long interval, TimeUnit unit) {
        executorService.scheduleWithFixedDelay(new PushTask(shard, supplier, unit.toMillis(interval)), 1, interval, unit);
    }

    @Override
    public void close() throws Exception {
        executorService.shutdownNow();
        httpClient.close();
    }

    private class PushTask implements Runnable {
        private final MetricFormat format = MetricFormat.SPACK;
        private final ShardKey shard;
        private final MetricSupplier supplier;
        private final long intervalMillis;

        public PushTask(ShardKey shard, MetricSupplier supplier, long intervalMillis) {
            this.shard = shard;
            this.supplier = supplier;
            this.intervalMillis = intervalMillis;
        }

        @Override
        public void run() {
            CompletableFutures.safeCall(() -> httpClient.executeRequest(new RequestBuilder(HttpConstants.Methods.POST)
                    .setUrl("https://solomon.yandex.net/api/v2/push"
                            + "?project=" + shard.getProject()
                            + "&cluster=" + shard.getCluster()
                            + "&service=" + shard.getService())
                    .setBody(encode())
                    .addHeader(CONTENT_TYPE.toString(), format.contentType())
                    .addHeader(AUTHORIZATION.toString(), "OAuth " + System.getProperty("OAuthToken"))
                    .build())
                    .toCompletableFuture())
                    .whenComplete((response, throwable) -> {
                        if (throwable != null) {
                            logger.warn("failed push to shard {}", shard, throwable);
                            return;
                        }

                        if (!HttpStatusClass.SUCCESS.contains(response.getStatusCode())) {
                            logger.warn("failed push to shard {}: {}" + shard, response);
                        }
                    });
        }

        private byte[] encode() {
            long gen = System.currentTimeMillis() / intervalMillis;
            long tsMillis = gen * intervalMillis;
            ByteArrayOutputStream out = new ByteArrayOutputStream(8 << 10); // 8 KiB
            try (MetricEncoder encoder = createEncoder(out, format, CompressionAlg.ZSTD)) {
                var consumer = new RateToCounterReplaceConsumer(encoder);
                consumer.onStreamBegin(supplier.estimateCount());
                consumer.onCommonTime(tsMillis);
                consumer.onLabelsBegin(1);
                consumer.onLabel("host", HostUtils.getShortName());
                consumer.onLabelsEnd();
                supplier.append(0, Labels.empty(), consumer);
                consumer.onStreamEnd();
            } catch (Exception e) {
                throw new IllegalStateException("cannot encode metrics", e);
            }
            return out.toByteArray();
        }
    }

    private static class RateToCounterReplaceConsumer implements MetricConsumer {
        private final MetricConsumer target;

        public RateToCounterReplaceConsumer(MetricConsumer target) {
            this.target = target;
        }

        @Override
        public void onStreamBegin(int countHint) {
            target.onStreamBegin(countHint);
        }

        @Override
        public void onStreamEnd() {
            target.onStreamEnd();
        }

        @Override
        public void onCommonTime(long tsMillis) {
            target.onCommonTime(tsMillis);
        }

        @Override
        public void onMetricBegin(MetricType type) {
            if (type == MetricType.RATE) {
                type = MetricType.COUNTER;
            }

            if (type == MetricType.HIST_RATE) {
                type = MetricType.HIST;
            }

            target.onMetricBegin(type);
        }

        @Override
        public void onMetricEnd() {
            target.onMetricEnd();
        }

        @Override
        public void onLabelsBegin(int countHint) {
            target.onLabelsBegin(countHint);
        }

        @Override
        public void onLabelsEnd() {
            target.onLabelsEnd();
        }

        @Override
        public void onLabel(Label label) {
            target.onLabel(label);
        }

        @Override
        public void onDouble(long tsMillis, double value) {
            target.onDouble(tsMillis, value);
        }

        @Override
        public void onLong(long tsMillis, long value) {
            target.onLong(tsMillis, value);
        }

        @Override
        public void onHistogram(long tsMillis, HistogramSnapshot snapshot) {
            target.onHistogram(tsMillis, snapshot);
        }
    }
}
