package ru.yandex.solomon.tool.migration.kv;

import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Throwables;
import com.google.common.net.HostAndPort;
import io.netty.channel.EventLoopGroup;
import org.apache.commons.io.FileUtils;

import ru.yandex.kikimr.client.KikimrGrpcTransport;
import ru.yandex.kikimr.client.KikimrTransport;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvClientImpl;
import ru.yandex.kikimr.client.kv.KikimrKvClientSync;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.misc.cmdline.CmdArgsChief;
import ru.yandex.misc.cmdline.CmdLineArgs;
import ru.yandex.misc.cmdline.Parameter;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.thread.factory.DaemonThreadFactory;
import ru.yandex.misc.thread.factory.ThreadNameThreadFactory;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.main.logger.LoggerConfigurationUtils;
import ru.yandex.solomon.selfmon.ng.JvmMon;
import ru.yandex.solomon.selfmon.ng.ProcSelfMon;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.tool.migration.kv.StockpileShardTransferMetrics.ShardMetrics;
import ru.yandex.solomon.tool.stockpile.backup.BackupsHelper;
import ru.yandex.solomon.util.ExceptionUtils;
import ru.yandex.solomon.util.NettyUtils;
import ru.yandex.solomon.util.PropertyInitializer;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.host.HostUtils;
import ru.yandex.solomon.util.time.DurationUtils;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.kikimrKv.KvChannel;
import ru.yandex.stockpile.kikimrKv.KvChannels;
import ru.yandex.stockpile.kikimrKv.KvTabletsMapping;
import ru.yandex.stockpile.server.data.names.FileKind;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.file.IndexFile;
import ru.yandex.stockpile.server.shard.KvLock;

/**
 * @author Maksim Leonov (nohttp@)
 */
@ParametersAreNonnullByDefault
public class StockpileShardTransfer {
    private static final RetryConfig retryConfig = RetryConfig.DEFAULT
            .withNumRetries(30)
            .withDelay(300)
            .withStats((timeSpentMillis, cause) -> {
                System.out.println("Failed, retrying...\n" + Throwables.getStackTraceAsString(cause));
            });

    private static final long TS = Instant.parse("2014-01-01T00:00:00Z").toEpochMilli() / 1000;
    private static final FileNamePrefix.TempBackup PREFIX = new FileNamePrefix.TempBackup(new FileNamePrefix.Backup(TS));
    private static final int MAX_FILES_IN_FLIGHT = 15;

    private final CmdlineArgs args;
    private final ExecutorService executor;
    private final EventLoopGroup io;
    private final ContextToConnectToKikimr sourceContext;
    private final ContextToConnectToKikimr targetContext;
    private final StockpileShardTransferMetrics metrics;

    public StockpileShardTransfer(CmdlineArgs args) {
        this.args = args;
        if (EnumSet.of(SolomonCluster.PROD_STORAGE_SAS, SolomonCluster.PROD_STORAGE_VLA).contains(args.targetClusterId)) {
            throw new RuntimeException("Dangerous");
        }

        MetricRegistry registry = new MetricRegistry(Labels.of("host", HostUtils.getShortName()));
        metrics = new StockpileShardTransferMetrics(registry
                .subRegistry("from", this.args.sourceClusterId.name())
                .subRegistry("to", this.args.targetClusterId.name()));
        JvmMon.addAllMetrics(registry);
        ProcSelfMon.addCpuTimeMetrics(registry);
        ProcSelfMon.addMemoryMetrics(registry);
        ProcSelfMon.addThreadsMetrics(registry);
        MetricsPushClient.create().schedulePush(
                new ShardKey("solomon", "push", "push"),
                registry);

        executor = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors() * 2,
                new DaemonThreadFactory(new ThreadNameThreadFactory(MetabaseTransfer.class)));
        io = NettyUtils.createEventLoopGroup(this.getClass().getSimpleName(), 4);
        JvmMon.addExecutorMetrics("common", executor, registry);
        sourceContext = new ContextToConnectToKikimr(this.args.sourceClusterId);
        targetContext = new ContextToConnectToKikimr(this.args.targetClusterId);
    }

    public static void main(String[] args) {
        try {
            PropertyInitializer.init();
            LoggerConfigurationUtils.disableLogger();
            CmdlineArgs parsed = CmdArgsChief.parse(CmdlineArgs.class, args);
            new StockpileShardTransfer(parsed).run();
            TimeUnit.SECONDS.sleep(30);
            System.exit(0);
        } catch (Throwable e) {
            ExceptionUtils.uncaughtException(e);
        }
    }

    private void run() {
        int[] shards;
        if (args.shards.isEmpty()) {
            throw new RuntimeException("Configure shards list!");
        } else if (args.shards.size() == 1) {
            targetContext.mapping.waitForReady();
            int[] allShards = targetContext.mapping.getShardIdStream().toArray();
            switch (args.shards.get(0)) {
                case "all":
                    shards = allShards;
                    break;
                case "local":
                    shards = getTargetShardsOnLocalhost(args, targetContext, allShards);
                    break;
                default:
                    shards = new int[]{Integer.parseInt(args.shards.get(0))};
                    break;
            }
        } else {
            shards = args.shards.stream().mapToInt(Integer::parseInt).toArray();
        }

        metrics.shardsCount.set(shards.length);
        for (int shardId : shards) {
            StockpileShardId id = new StockpileShardId(shardId);
            System.out.println("Processing stockpile shard: " + shardId);
            args.mode.worker.run(this, id);
            metrics.shardsProcessed.inc();
        }
    }

    private int[] getTargetShardsOnLocalhost(CmdlineArgs cc, ContextToConnectToKikimr targetContext, int[] allShards) {
        if (!(cc.targetClusterId.hosts().contains(HostUtils.getFqdn()))) {
            throw new RuntimeException("`local' selector can only be used on a machine which is a part of `target' cluster");
        }
        Set<Long> tabletsOnLocalhost = new HashSet<>();
        // TODO: this one is incorrect; it searches for the list of tablets on the machine we're connected to, which is not necessary a localhost
        for (long localTablet : CompletableFutures.join(targetContext.kvClient.findTabletsOnLocalhost())) {
            tabletsOnLocalhost.add(localTablet);
        }
        System.out.println("fqdn: " + HostUtils.getFqdn());
        System.out.println("Tablets on localhost: " + tabletsOnLocalhost.size());
        return IntStream.of(allShards)
                .filter(shardId -> {
                    long kvTabletId = targetContext.mapping.getTabletId(shardId);
                    boolean shardIsLocal = tabletsOnLocalhost.contains(kvTabletId);
                    if (shardIsLocal) {
                        System.out.println("Local shard: " + StockpileShardId.toString(shardId) + " -> " + kvTabletId);
                        return true;
                    }

                    return false;
                })
                .toArray();
    }

    private void makeStamps(StockpileShardId shardId) {
        long targetTabletId = targetContext.mapping.getTabletId(shardId.getId());

        System.out.println("Target tablet: " + targetTabletId);
        KvLock lock = new KvLock(
            HostUtils.getFqdn(),
            ProcessHandle.current().pid(),
            Instant.now().plus(1, ChronoUnit.DAYS).getEpochSecond());

        targetContext.kvClient.write(
                targetTabletId, 0,
                KvLock.FILE, KvLock.serialize(lock),
                KvChannel.HDD.getChannel(), MsgbusKv.TKeyValueRequest.EPriority.REALTIME,
            0).join();
    }

    private void makeLargeStamps(StockpileShardId shardId) {
        long targetTabletId = targetContext.mapping.getTabletId(shardId.getId());

        System.out.println("Target tablet: " + targetTabletId);
        KvLock lock = new KvLock(
            HostUtils.getFqdn(),
            ProcessHandle.current().pid(),
            Instant.now().plus(1, ChronoUnit.DAYS).getEpochSecond());

        AtomicInteger chunk = new AtomicInteger();

        withRetriesRunnable(
                () -> targetContext.kvClientSync.writeLarge(
                        targetTabletId, 0,
                        KvLock.FILE, KvLock.serialize(lock),
                        () -> "chunk." + chunk.incrementAndGet(),
                        KvChannel.HDD.getChannel(), MsgbusKv.TKeyValueRequest.EPriority.REALTIME
                )
        ).join();
    }

    private void removeStamps(StockpileShardId shardId) {
        long targetTabletId = targetContext.mapping.getTabletId(shardId.getId());
        removeLockFile(targetContext, targetTabletId);
    }

    private void removeAll(StockpileShardId shardId) {
        long targetTabletId = targetContext.mapping.getTabletId(shardId.getId());
        // 1. clean up target kv (everything except stamp)
        List<String> filesToDelete = new ArrayList<>();

        var entries = withRetries(() -> targetContext.kvClient.readRangeNames(targetTabletId, 0, 0)).join();
        for (KikimrKvClient.KvEntryStats kvEntryStats : entries) {
            filesToDelete.add(kvEntryStats.getName());
        }

        withRetries(() -> targetContext.kvClient.deleteFiles(targetTabletId, 0, filesToDelete, 0)).join();
    }

    private void checkStamps(StockpileShardId shardId) {
        long targetTabletId = targetContext.mapping.getTabletId(shardId.getId());
        var result = targetContext.kvClientSync.readData(targetTabletId, 0, KvLock.FILE);
        if (result.isEmpty()) {
            System.out.println("No stamp on shard " + shardId);
        }

        KvLock lock = KvLock.parse(result.get());
        if (lock.isExpired()) {
            System.out.println(lock + " expired on shard " + shardId);
        }
    }

    private void validateLs(StockpileShardId shardId) {
        long sourceTabletId = sourceContext.mapping.getTabletId(shardId.getId());
        withRetries(
                () -> sourceContext.backupHelper.backup(sourceTabletId, 0, FileNamePrefix.Current.instance, PREFIX)
        ).join();

        checkBackupBug(shardId, sourceContext, sourceTabletId);

        withRetriesRunnable(
                () -> sourceContext.backupHelper.deleteBackups(sourceTabletId, 0, PREFIX)
        ).join();
    }

    private boolean checkBackupBug(StockpileShardId shardId, ContextToConnectToKikimr sourceContext, long sourceTabletId) {
        List<KikimrKvClient.KvEntryStats> files = withRetriesSync(
                () -> sourceContext.backupHelper.listFilesInBackupSync(sourceTabletId, 0, PREFIX)
        ).join();

        List<IndexFile> indices = files.stream()
                .map(e -> FileNameParsed.parseInBackup(PREFIX, e.getName()))
                .filter(file -> file.fileKind() == FileKind.DAILY_INDEX || file.fileKind() == FileKind.ETERNITY_INDEX)
                .map(file -> (IndexFile) file)
                .collect(Collectors.toList());

        long dailyIndexCount = indices.stream()
                .filter(file -> file.fileKind() == FileKind.DAILY_INDEX)
                .mapToLong(IndexFile::txn)
                .distinct().count();

        long eternityIndexCount = indices.stream()
                .filter(file -> file.fileKind() == FileKind.ETERNITY_INDEX)
                .mapToLong(IndexFile::txn)
                .distinct().count();

        if (dailyIndexCount > 1 || eternityIndexCount > 1) {
            System.out.println("Shard " + shardId + ": suspicious backup was detected! " + new Date());

            List<KikimrKvClient.KvEntryStats> realFiles = withRetries(
                    () -> sourceContext.kvClient.readRangeNames(sourceTabletId, 0, 0)
            ).join();

            realFiles.forEach(file -> System.out.println(file.getName() + " " + Date.from(Instant.ofEpochSecond(file.getCreatedUnixtime()))));
            return true;
        } else {
            return false;
        }
    }

    private void migrate(StockpileShardId shardId) {
        new MigrationTask(shardId).run();
    }

    private CompletableFuture<Void> withRetriesRunnable(Runnable supplier) {
        return withRetries(() -> CompletableFuture.runAsync(supplier));
    }

    private <T> CompletableFuture<T> withRetriesSync(Supplier<T> supplier) {
        return withRetries(() -> CompletableFuture.supplyAsync(supplier));
    }

    private <T> CompletableFuture<T> withRetries(Supplier<CompletableFuture<T>> supplier) {
        return RetryCompletableFuture.runWithRetries(
                supplier,
                retryConfig
        );
    }

    private MsgbusKv.TKeyValueRequest.EStorageChannel getChannel(String nameInPrefix) {
        return FileNameParsed.parseCurrentOptional(nameInPrefix)
                .map(fileNameParsed -> KvChannels.byFileKind(fileNameParsed.fileKind()))
                .orElse(KvChannel.HDD)
                .getChannel();
    }

    private void removeLockFile(ContextToConnectToKikimr targetContext, long targetTabletId) {
        withRetries(() -> targetContext.kvClient.deleteFiles(targetTabletId, 0, List.of(KvLock.FILE), 0)).join();
    }

    enum Mode {
        MAKE_STAMPS(StockpileShardTransfer::makeStamps),
        VALIDATE_BACKUP(StockpileShardTransfer::validateLs),
        MAKE_LARGE_STAMPS(StockpileShardTransfer::makeLargeStamps),
        DELETE_STAMPS(StockpileShardTransfer::removeStamps),
        DELETE_ALL(StockpileShardTransfer::removeAll),
        CHECK_STAMPS(StockpileShardTransfer::checkStamps),
        MIGRATE(StockpileShardTransfer::migrate),
        ;

        private final Worker worker;

        Mode(Worker worker) {
            this.worker = worker;
        }
    }

    private interface Worker {
        void run(StockpileShardTransfer context, StockpileShardId shardId);
    }

    @CmdLineArgs
    public static class CmdlineArgs {
        @Parameter(names = "--source", description = "Source stockpile cluster id")
        public SolomonCluster sourceClusterId;
        @Parameter(names = "--target", description = "Source stockpile cluster id")
        public SolomonCluster targetClusterId;

        @Parameter(names = "--mode", description = "What to do")
        public Mode mode = Mode.MIGRATE;
        @Parameter(description = "stockpile shards to migrate")
        public List<String> shards;
    }

    private class ContextToConnectToKikimr {
        SolomonCluster cluster;
        KikimrKvClient kvClient;
        KikimrKvClientSync kvClientSync;
        BackupsHelper backupHelper;
        KvTabletsMapping mapping;

        ContextToConnectToKikimr(SolomonCluster cluster) {
            this.cluster = cluster;
            KikimrTransport transport = new KikimrGrpcTransport(
                    getKikimrAdresses(cluster),
                    26 << 20,
                    Duration.ofSeconds(15),
                    Duration.ofSeconds(180),
                    io,
                    executor);

            this.kvClient = new KikimrKvClientImpl(transport);
            this.kvClientSync = new KikimrKvClientSync(kvClient);
            this.backupHelper = new BackupsHelper(kvClient);
            ScheduledExecutorService executor = Executors.newScheduledThreadPool(4);
            if (!EnumSet.of(SolomonCluster.PROD_STORAGE_SAS, SolomonCluster.PROD_STORAGE_VLA).contains(cluster)) {
                this.mapping = new KvTabletsMapping("", kvClient, executor, executor);
            } else {
                this.mapping = new KvTabletsMapping(cluster.getSolomonVolumePath(), kvClient, executor, executor);
            }
            this.mapping.waitForReady();
        }

        private List<HostAndPort> getKikimrAdresses(SolomonCluster cluster) {
            if (cluster.hosts().contains(HostUtils.getFqdn())) {
                return Collections.singletonList(HostAndPort.fromParts("localhost", 2135));
            }

            return cluster.addressesKikimrGrpc();
        }
    }

    private class MigrationTask implements Runnable {
        private final StockpileShardId shardId;
        private final ShardMetrics shardMetrics;
        private final long sourceTabletId;
        private final long targetTabletId;

        public MigrationTask(StockpileShardId shardId) {
            this.shardId = shardId;
            this.shardMetrics = metrics.getShardMetrics(shardId.getId());
            this.shardMetrics.startedAtNanos = System.nanoTime();
            this.sourceTabletId = sourceContext.mapping.getTabletId(shardId.getId());
            this.targetTabletId = targetContext.mapping.getTabletId(shardId.getId());
        }

        @Override
        public void run() {
            try {
                cleanUpTargetKv();
                forceSourceSnapshot();
                forceSourceBackup();
                copyData();
                checkFileLists();
                removeLockFile();
                shardMetrics.completedAtNanos = System.nanoTime();
                System.out.println(msgPrefix() + " successful migrated");
            } finally {
               deleteSourceBackups();
            }
        }

        private String msgPrefix() {
            return shardId + " (" + timeSpend() + "): ";
        }

        private String timeSpend() {
            long spendNanos = System.nanoTime() - shardMetrics.startedAtNanos;
            long spendMillis = TimeUnit.NANOSECONDS.toMillis(spendNanos);
            if (spendMillis < 1000) {
                return spendMillis + "ms";
            }
            return DurationUtils.formatDurationMillis(spendMillis);
        }

        private void cleanUpTargetKv() {
            // 1. clean up target kv (everything except stamp)
            List<String> filesToDelete = new ArrayList<>();
            boolean hasStamp = false;

            var entries = withRetries(() -> targetContext.kvClient.readRangeNames(targetTabletId, 0, 0)).join();
            for (KikimrKvClient.KvEntryStats kvEntryStats : entries) {
                if (kvEntryStats.getName().equals(KvLock.FILE)) {
                    hasStamp = true;
                } else {
                    filesToDelete.add(kvEntryStats.getName());
                }
            }

            if (!hasStamp) {
                throw new RuntimeException("Target shard " + shardId + " has no stamp!");
            }

            long count = filesToDelete.size();
            withRetries(() -> targetContext.kvClient.deleteFiles(targetTabletId, 0, filesToDelete, 0)).join();
            System.out.println(msgPrefix() + " rm files " + count + " from " + targetContext.cluster);
        }

        private void forceSourceSnapshot() {
            // TODO. BTW, this step is optional
        }

        private void forceSourceBackup() {
            while (true) {
                withRetries(
                        () -> sourceContext.backupHelper.backup(sourceTabletId, 0, FileNamePrefix.Current.instance, PREFIX)
                ).join();
                System.out.println(msgPrefix() + " backup created on " + sourceContext.cluster);

                // check for a bug with multiple daily snapshots...
                if (checkBackupBug(shardId, sourceContext, sourceTabletId)) {
                    System.out.println(msgPrefix() + " backup contains bug, retry create");
                    deleteSourceBackups();
                } else {
                    return;
                }
            }
        }

        private void deleteSourceBackups() {
            withRetriesRunnable(
                    () -> sourceContext.backupHelper.deleteBackups(sourceTabletId, 0, PREFIX)
            ).join();
            System.out.println(msgPrefix() + " backup deleted from " + sourceContext.cluster);
        }

        private void copyData() {
            List<KikimrKvClient.KvEntryStats> files = withRetriesSync(
                    () -> sourceContext.backupHelper.listFilesInBackupSync(sourceTabletId, 0, PREFIX)
            ).join();
            shardMetrics.sourceFileCount.set(files.size());
            long expectedBytes = files.stream().mapToLong(KikimrKvClient.KvEntryStats::getSize).sum();
            shardMetrics.sourceFileBytes.set(expectedBytes);

            System.out.println(msgPrefix() + FileUtils.byteCountToDisplaySize(expectedBytes) +" to migrate");

            AtomicLong totalBytes = new AtomicLong();
            AtomicInteger cursor = new AtomicInteger();

            long startMillis = System.currentTimeMillis();
            AsyncActorBody body = () -> {
                int index = cursor.getAndIncrement();
                if (index == files.size()) {
                    return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
                }

                KikimrKvClient.KvEntryStats file = files.get(index);
                return withRetries(() -> readAndWrite(file))
                        .whenComplete((unit, throwable) -> {
                            if (throwable == null) {
                                totalBytes.addAndGet(file.getSize());
                                shardMetrics.targetFileBytes.add(file.getSize());
                                shardMetrics.targetFileCount.inc();
                                long percent = Math.round(((double) shardMetrics.targetFileCount.get() / (double) shardMetrics.sourceFileCount.get()) * 100.0);
                                System.err.println(msgPrefix() + " files " + file.getName() + " migrated " + percent + "%");
                            }
                        });
            };

            AsyncActorRunner actorRunner = new AsyncActorRunner(body, executor, MAX_FILES_IN_FLIGHT);
            actorRunner.start().join();
            long dtMillis = System.currentTimeMillis() - startMillis;
            System.out.println(msgPrefix()
                    + " moved "
                    + FileUtils.byteCountToDisplaySize(totalBytes.get())
                    + " in "
                    + DurationUtils.formatDurationMillis(dtMillis));
        }

        private void checkFileLists() {
            Set<String> expectedFileNames = new HashSet<>();
            expectedFileNames.add(KvLock.FILE);

            List<KikimrKvClient.KvEntryStats> sourceFiles = withRetriesSync(
                    () -> sourceContext.backupHelper.listFilesInBackupSync(sourceTabletId, 0, PREFIX)
            ).join();

            for (KikimrKvClient.KvEntryStats sourceFile : sourceFiles) {
                String nameInPrefix = sourceFile.getName().replace(PREFIX.format(), "");
                String nameInCurrent = FileNamePrefix.Current.instance.format() + nameInPrefix;
                expectedFileNames.add(nameInCurrent);
            }

            List<KikimrKvClient.KvEntryStats> targetFiles = withRetries(
                    () -> targetContext.kvClient.readRangeNames(targetTabletId, 0, 0)
            ).join();

            for (KikimrKvClient.KvEntryStats targetFile : targetFiles) {
                if (!expectedFileNames.remove(targetFile.getName())) {
                    throw new RuntimeException("Unexpected file: " + targetFile.getName());
                }
            }
            if (!expectedFileNames.isEmpty()) {
                throw new RuntimeException("Something is missing: " + expectedFileNames.toString());
            }
            System.out.println(msgPrefix() + " successful checked files into backup and copied");
        }

        private void removeLockFile() {
            withRetries(() -> targetContext.kvClient.deleteFiles(targetTabletId, 0, List.of(KvLock.FILE), 0))
                    .join();
            System.out.println(msgPrefix() + " migration stamps deleted");
        }

        private CompletableFuture<Void> readAndWrite(KikimrKvClient.KvEntryStats file) {
            String nameInPrefix = file.getName().replace(PREFIX.format(), "");

            String nameInCurrent = FileNamePrefix.Current.instance.format() + nameInPrefix;

            MsgbusKv.TKeyValueRequest.EStorageChannel channel = getChannel(nameInCurrent);

            CompletableFuture<Void> result = new CompletableFuture<>();
            CompletableFuture<byte[]> readFuture = sourceContext.kvClient.readDataLarge(
                sourceTabletId, 0, file.getName(), 0, MsgbusKv.TKeyValueRequest.EPriority.BACKGROUND);
            metrics.reads.forFuture(readFuture);
            readFuture.whenComplete((bytes, throwable) -> {
                if (throwable != null) {
                    result.completeExceptionally(throwable);
                } else {
                    try {
                        CompletableFuture<Void> writeFuture;
                        if (bytes.length < KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE) {
                            writeFuture = targetContext.kvClient.write(
                                    targetTabletId, 0, nameInCurrent, bytes, channel, MsgbusKv.TKeyValueRequest.EPriority.REALTIME, 0);
                        } else {
                            AtomicInteger chunkSuffix = new AtomicInteger();
                            targetContext.kvClientSync.writeLarge(
                                    targetTabletId, 0, nameInCurrent, bytes,
                                    () -> StockpileKvNames.TMP_PREFIX + nameInCurrent + "." + chunkSuffix.getAndIncrement(),
                                    channel, MsgbusKv.TKeyValueRequest.EPriority.REALTIME
                            );
                            writeFuture = CompletableFuture.completedFuture(null);
                        }

                        metrics.writes.forFuture(writeFuture);
                        writeFuture
                                .whenComplete((writeResp, e2) -> {
                                    if (e2 != null) {
                                        result.completeExceptionally(e2);
                                    } else {
                                        try {
                                            result.complete(writeResp);
                                        } catch (Exception e) {
                                            result.completeExceptionally(e);
                                        }
                                    }
                                });
                    } catch (Exception e) {
                        result.completeExceptionally(e);
                    }
                }
            });
            return result;
        }
    }
}
