package ru.yandex.solomon.tool.migration.kv;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;

import ru.yandex.monlib.metrics.primitives.Counter;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.LazyGaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileShardTransferMetrics {
    private final MetricRegistry registry;
    private final ConcurrentMap<Integer, ShardMetrics> metrics = new ConcurrentHashMap<>();

    final GaugeInt64 shardsCount;
    final Counter shardsProcessed;
    final LazyGaugeInt64 shardsWaiting;
    final AsyncMetrics reads;
    final AsyncMetrics writes;


    public StockpileShardTransferMetrics(MetricRegistry registry) {
        this.registry = registry;
        this.reads = new AsyncMetrics(registry, "kv.source.read.");
        this.writes = new AsyncMetrics(registry, "kv.target.write.");
        this.shardsCount = registry.gaugeInt64("kv.shard.count");
        this.shardsProcessed = registry.counter("kv.shard.processed");
        this.shardsWaiting = registry.lazyGaugeInt64("kv.shard.waiting", () -> shardsCount.get() - shardsProcessed.get());
    }

    public ShardMetrics getShardMetrics(int shardId) {
        return this.metrics.computeIfAbsent(shardId, id -> new ShardMetrics(id.toString(), registry));
    }

    public static class ShardMetrics {
        final GaugeInt64 sourceFileCount;
        final GaugeInt64 sourceFileBytes;
        final Counter targetFileCount;
        final Counter targetFileBytes;

        volatile long startedAtNanos;
        volatile long completedAtNanos;

        public ShardMetrics(String shardId, MetricRegistry registry) {
            this(registry.subRegistry("shardId", shardId));
        }

        public ShardMetrics(MetricRegistry registry) {
            sourceFileCount = registry.gaugeInt64("kv.source.files.count");
            sourceFileBytes = registry.gaugeInt64("kv.source.files.bytes");
            targetFileCount = registry.counter("kv.target.files.count");
            targetFileBytes = registry.counter("kv.target.files.bytes");

            registry.lazyCounter("kv.shard.migration.timeMs", () -> {
                if (startedAtNanos == 0) {
                    return 0;
                }

                if (completedAtNanos == 0) {
                    return TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startedAtNanos);
                }

                return TimeUnit.NANOSECONDS.toMillis(completedAtNanos - startedAtNanos);
            });
        }
    }
}
