package ru.yandex.solomon.tool.stockpile;

import java.time.Instant;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Optional;

import com.google.common.base.Joiner;

import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.tool.stockpile.backup.BackupWithStats;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileBackupBackup extends StockpileBackupCommand {

    private final String path;

    private StockpileBackupBackup(SolomonCluster cluster, String path) {
        super(cluster);
        this.path = path;
    }

    public static StockpileBackupCommand of(String[] args) {
        if (args.length != 1) {
            printUsage();
            System.exit(1);
        }

        SolomonCluster cluster = SolomonCluster.valueOf(args[0]);
        return new StockpileBackupBackup(cluster, cluster.getSolomonVolumePath());
    }

    @Override
    public void run() {
        System.out.println("create kv tablets...");
        long[] tabletIds = toolCommon.resolveKvTabletsForShards(path);
        System.out.println("create kv tablets done.");

        LinkedHashMap<FileNamePrefix.Backup, List<BackupWithStats>> backupsByBackup =
                backupsHelper.listBackups(tabletIds);

        if (backupsByBackup.size() > 1) {
            System.err.println("not a single backup:\n" + Joiner.on("\n").join(backupsByBackup.keySet()));
            System.exit(1);
        }

        Optional<FileNamePrefix.Backup> prevBackup = backupsByBackup.keySet()
                .stream()
                .sorted(Comparator.comparingLong(FileNamePrefix.Backup::getInstantSeconds).reversed())
                .findFirst();

        if (prevBackup.isPresent()) {
            System.out.println("prev backup: " + prevBackup.get());

            if (prevBackup.get().getInstantSeconds() > InstantUtils.currentTimeSeconds() - 12 * 3600) {
                System.err.println("previous backup is made later that 10 hours ago, at: " + Instant.ofEpochSecond(prevBackup.get().getInstantSeconds()));
                System.exit(1);
            }
        }

        FileNamePrefix.Backup backup = new FileNamePrefix.Backup(Instant.now().getEpochSecond());
        System.out.println("backup prefix: " + backup);
        toolCommon.stockpileForEachShard(tabletIds, tabletId -> backupsHelper.backupCurrent(tabletId, 0, backup));
        System.out.println("backup done: " + backup);
    }

    public static void printUsage() {
        System.err.println("Usage: tool backup <cluster_id>");
    }
}
