package ru.yandex.solomon.tool.stockpile;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import com.google.protobuf.ByteString;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Verify;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveNakedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.metrics.client.TimeSeriesCodec;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.TimeSeries;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.tool.StockpileHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.shard.StockpileMetricId;


/**
 * @author Sergey Polovko
 */
public class StockpileCopyMetricData {
    private static final ZoneId MOSCOW = ZoneId.of("Europe/Moscow");

    private static final class CliArgs {
        final int solomonShardNum;
        final SolomonCluster fromCluster;
        final SolomonCluster toCluster;
        final StockpileMetricId fromId;
        final StockpileMetricId toId;
        final long fromMillis;
        final long toMillis;

        private CliArgs(
            int solomonShardNum,
            SolomonCluster fromCluster, SolomonCluster toCluster,
            StockpileMetricId fromId, StockpileMetricId toId,
            long fromMillis, long toMillis)
        {
            this.solomonShardNum = solomonShardNum;
            this.fromCluster = fromCluster;
            this.toCluster = toCluster;
            this.fromId = fromId;
            this.toId = toId;
            this.fromMillis = fromMillis;
            this.toMillis = toMillis;
        }

        public static CliArgs parse(String[] args) {
            String[] from = StringUtils.split(args[1], ':');
            String[] to = StringUtils.split(args[2], ':');

            return new CliArgs(
                Integer.parseInt(args[0]),
                SolomonCluster.valueOf(from[0]),
                SolomonCluster.valueOf(to[0]),
                StockpileMetricId.parse(from[1]),
                StockpileMetricId.parse(to[1]),
                LocalDateTime.parse(args[3]).atZone(MOSCOW).toEpochSecond() * 1000,
                LocalDateTime.parse(args[4]).atZone(MOSCOW).toEpochSecond() * 1000
            );
        }
    }

    public static void main(String[] args) throws InterruptedException {
        if (args.length != 5) {
            System.err.println(
                "Usage:\n" +
                "    java " + StockpileCopyMetricData.class + " <solomonShardNum> <fromId> <toId> <fromTime> <toTime>\n" +
                "\n" +
                "Where:\n" +
                "    * fromId/toId       - <clusterId>:<shardId>/<localId>\n" +
                "    * fromTime/toTime   - MSK date time in format 'yyyy-MM-ddTHH:mm:ss'\n" +
                "\n" +
                "Example:\n" +
                "    $ java ru.yandex.solomon.tool.stockpile.StockpileCopyMetricData \\\n" +
                "            2055 \\\n" +
                "            PROD_STP_MYT:2808/2467908839705627808 \\\n" +
                "            PROD_STP_MAN:3433/2521035401309305537 \\\n" +
                "            2018-03-29T16:12 \\\n" +
                "            2018-03-29T16:35\n");
            System.exit(1);
        }

        CliArgs cliArgs = CliArgs.parse(args);
        Verify.isTrue(cliArgs.fromCluster != cliArgs.toCluster, "clusters must not be equal");

        ScheduledExecutorService executor = Executors.newScheduledThreadPool(4);
        try {
            StockpileClient stockpileFrom = StockpileHelper.createGrpcClient(cliArgs.fromCluster, executor);
            StockpileClient stockpileTo = StockpileHelper.createGrpcClient(cliArgs.toCluster, executor);

            System.err.println("init Stockpile clients...");
            CompletableFuture.allOf(stockpileFrom.forceUpdateClusterMetaData(), stockpileTo.forceUpdateClusterMetaData()).join();
            // XXX: unfortunately we need this hack
            TimeUnit.SECONDS.sleep(2);
            System.err.println("init Stockpile clients [DONE]");

            var data = readMetricData(
                stockpileFrom,
                cliArgs.fromId,
                cliArgs.fromMillis, cliArgs.toMillis);

            writeMetricData(stockpileTo, cliArgs.solomonShardNum, cliArgs.toId, data);

            stockpileFrom.close();
            stockpileTo.close();
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        } finally {
            executor.shutdownNow();
            executor.awaitTermination(5, TimeUnit.SECONDS);
        }

        System.exit(0);
    }

    private static AggrGraphDataIterable readMetricData(
        StockpileClient client,
        StockpileMetricId metricId,
        long fromMillis, long toMillis)
    {
        System.err.println("read metric data...");
        var format = StockpileFormat.byNumberOrCurrent(client.getCompatibleCompressFormat().upperEndpoint());
        var response = client.readCompressedOne(TReadRequest.newBuilder()
            .setMetricId(MetricId.newBuilder()
                .setShardId(metricId.getShardId())
                .setLocalId(metricId.getLocalId())
                .build())
            .setFromMillis(fromMillis)
            .setToMillis(toMillis)
            .setBinaryVersion(format.getFormat())
            .build())
            .join();

        if (response.getStatus() != EStockpileStatusCode.OK) {
            throw new StockpileRuntimeException(response.getStatus(), response.getStatusMessage());
        }

        var result = TimeSeriesCodec.sequenceDecode(response);
        System.err.println("read metric data [DONE]");
        return result;
    }

    private static void writeMetricData(
        StockpileClient client,
        int solomonShardNum,
        StockpileMetricId metricId,
        AggrGraphDataIterable data)
    {
        System.err.println("write metric data...");

        var format = StockpileFormat.byNumberOrCurrent(client.getCompatibleCompressFormat().upperEndpoint());
        var archive = new MetricArchiveMutable(MetricHeader.defaultValue, format);
        archive.setOwnerShardId(solomonShardNum);
        archive.setOwnerProjectIdEnum(EProjectId.SOLOMON);
        archive.ensureCapacity(data.columnSetMask(), data.getRecordCount());

        AggrPoint point = new AggrPoint();
        var it = data.iterator();
        while (it.next(point)) {
            ZonedDateTime time = Instant.ofEpochMilli(point.tsMillis).atZone(MOSCOW);
            System.out.println(time.format(DateTimeFormatter.ISO_LOCAL_DATE_TIME) + "\t" + point.getValueDividedOrNull());

            archive.addRecordData(it.columnSetMask(), point);
        }

        var response = client.writeCompressedOne(prepare(metricId, archive)).join();
        if (response.getStatus() != EStockpileStatusCode.OK) {
            throw new StockpileRuntimeException(response.getStatus(), response.getStatusMessage());
        }

        System.err.println("write metric data [DONE]");
    }

    private static TCompressedWriteRequest prepare(StockpileMetricId metricId, MetricArchiveMutable archive) {
        return TCompressedWriteRequest.newBuilder()
            .setMetricId(MetricId.newBuilder()
                .setShardId(metricId.getShardId())
                .setLocalId(metricId.getLocalId())
                .build())
            .addChunks(serialize(archive))
            .setBinaryVersion(archive.getFormat().getFormat())
            .build();
    }

    private static TimeSeries.Chunk serialize(MetricArchiveMutable archive) {
        ByteString data = MetricArchiveNakedSerializer.serializerForFormatSealed(archive.getFormat())
            .serializeToByteString(archive.toImmutableNoCopy());

        return TimeSeries.Chunk.newBuilder()
            .setContent(data)
            .setPointCount(archive.getRecordCount())
            .build();
    }
}
