package ru.yandex.solomon.tool.stockpile;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvClientSync;
import ru.yandex.kikimr.client.kv.StringMicroUtils;
import ru.yandex.salmon.proto.StockpileCanonicalProto;
import ru.yandex.solomon.tool.KikimrHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.protobuf.ProtobufText;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.ser.test.convert.IndexConverters;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContentSerializer;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.IndexAddressWithFileCount;
import ru.yandex.stockpile.server.data.names.file.IndexFile;

/**
 * @author Stepan Koltsov
 */
public class StockpileDumpFile implements AutoCloseable {

    private final KikimrKvClientSync kikimrKvClientSync;
    private final StockpileToolCommon toolCommon;

    public StockpileDumpFile(SolomonCluster cluster) {
        KikimrKvClient kvClient = KikimrHelper.createKvClient(cluster);
        this.kikimrKvClientSync = new KikimrKvClientSync(kvClient);
        this.toolCommon = new StockpileToolCommon(kvClient);
    }

    public void run(String path, StockpileShardId shardId, String prefix) {
        long tabletId = toolCommon.resolveKvTabletsForShards(path)[shardId.getId() - 1];

        List<KikimrKvClient.KvEntryStats> names = kikimrKvClientSync.kvReadRangeNames(
                tabletId, 0, StringMicroUtils.asciiPrefixToRange(prefix));

        List<FileNameParsed> fileNames = names.stream()
            .map(e -> FileNameParsed.parseCurrent(e.getName()))
            .collect(Collectors.toList());

        IndexAddressWithFileCount indexName = IndexAddressWithFileCount.filter(fileNames).stream()
            .collect(CollectorsF.single());

        List<byte[]> parts = new ArrayList<>(indexName.getCount());
        for (IndexFile index : indexName.indexFileNames()) {
            String fileName = index.reconstructCurrent();
            parts.add(kikimrKvClientSync.readData(tabletId, 0, fileName).get());
        }

        SnapshotIndexContent indexContent = SnapshotIndexContentSerializer.S.deserializeParts(parts.toArray(new byte[0][]));
        StockpileCanonicalProto.Index indexProto = IndexConverters.toProto(indexContent);
        System.out.print(ProtobufText.serializeToTextEnsureTrailingLf(indexProto));
    }

    @Override
    public void close() {
        kikimrKvClientSync.close();
    }

    public static void main(String[] args) {
        if (args.length != 3) {
            System.out.println("Usage: tool <cluster> <shard> <prefix>");
            System.exit(1);
        }

        SolomonCluster cluster = SolomonCluster.valueOf(args[0]);
        StockpileShardId shardId = StockpileShardId.parseObject(args[1]);
        String prefix = args[2];

        try (StockpileDumpFile tool = new StockpileDumpFile(cluster)) {
            tool.run(cluster.getSolomonVolumePath(), shardId, prefix);
        }
    }
}
