package ru.yandex.solomon.tool.stockpile;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.solomon.tool.KikimrHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;
import ru.yandex.stockpile.server.data.names.IndexAddressWithFileCount;
import ru.yandex.stockpile.server.data.names.LogAddressWithFileCount;
import ru.yandex.stockpile.server.data.names.file.IndexFile;
import ru.yandex.stockpile.server.data.names.file.LogFile;

/**
 * @author Stepan Koltsov
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public class StockpileDumpLogsAndIndexes {

    public static void main(String[] args) throws IOException {
        if (args.length != 2) {
            System.err.println("usage: tool <cluster_id> <shard_id>");
            System.exit(1);
        }

        SolomonCluster cluster = SolomonCluster.valueOf(args[0]);
        StockpileShardId shardId = StockpileShardId.parseObject(args[1]);

        Path shardDir = Paths.get(shardId.toString());
        if (!Files.exists(shardDir)) {
            Files.createDirectories(shardDir);
        }

        try (KikimrKvClient kvClient = KikimrHelper.createKvClient(cluster)) {
            StockpileToolCommon tool = new StockpileToolCommon(kvClient);
            tool.withTempSnapshot2(cluster.getSolomonVolumePath(), shardId, FileNamePrefix.Current.instance, (tabletId, prefix, inBackupFileNameSet) -> {
                for (LogAddressWithFileCount log : inBackupFileNameSet.getLogs()) {
                    for (LogFile logFile : log.logFileNames()) {
                        Path filePath = filePath(shardDir, logFile);
                        if (Files.exists(filePath)) {
                            continue;
                        }

                        System.out.println("loading log file: " + logFile);
                        byte[] bytes = readKvFile(kvClient, tabletId, prefix, logFile);
                        writeFile(filePath, bytes);
                    }
                }

                for (IndexAddressWithFileCount index : inBackupFileNameSet.getIndexes()) {
                    for (IndexFile indexFile : index.indexFileNames()) {
                        Path filePath = filePath(shardDir, indexFile);
                        if (Files.exists(filePath)) {
                            continue;
                        }

                        System.out.println("loading index file: " + indexFile);
                        byte[] bytes = readKvFile(kvClient, tabletId, prefix, indexFile);
                        writeFile(filePath, bytes);
                    }
                }
            });
        } finally {
            System.exit(0);
        }
    }

    private static byte[] readKvFile(KikimrKvClient kvClient, long tabletId, FileNamePrefix prefix, FileNameParsed kvFileName) {
        return kvClient.readDataSome(tabletId, 0, kvFileName.reconstruct(prefix), 0, -1, 0, MsgbusKv.TKeyValueRequest.EPriority.BACKGROUND).join();
    }

    private static Path filePath(Path dir, FileNameParsed fileNameParsed) {
        String fileName = fileNameParsed.reconstructCurrent();
        return dir.resolve(Paths.get(fileName));
    }

    private static void writeFile(Path filePath, byte[] bytes) {
        try (OutputStream out = Files.newOutputStream(filePath)) {
            out.write(bytes);
        } catch (IOException e) {
            throw ExceptionUtils.throwException(e);
        }
    }
}
