package ru.yandex.solomon.tool.stockpile;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;

import com.google.common.base.Throwables;
import com.google.protobuf.ByteString;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveNakedSerializer;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.TimeSeries;
import ru.yandex.solomon.model.timeseries.decim.DecimPoliciesPredefined;
import ru.yandex.solomon.model.timeseries.decim.DecimatingAggrGraphDataIterator;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.api.EDecimPolicy;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TCommandRequest;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TShardCommandRequest;
import ru.yandex.stockpile.api.grpc.StockpileRuntimeException;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.shard.StockpileShardId;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileShardWriter {
    private static final int ARCHIVE_BYTES_SIZE_TO_FORCE_DECIM = 100 << 20; // 100 MiB;
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
        .withNumRetries(Integer.MAX_VALUE)
        .withDelay(TimeUnit.MINUTES.toMillis(1))
        .withStats((timeSpentMillis, cause) -> {
            MetricRegistry.root().counter("stockpile.shard.write.retry").inc();
            System.out.println("Failed, retrying...\n" + Throwables.getStackTraceAsString(cause));
        });

    private static final long BYTE_SIZE_LIMIT = (1 << 20) / 2; // 0.5 MiB

    private final int shardId;
    private final ConcurrentLinkedQueue<Metric> queue = new ConcurrentLinkedQueue<>();
    private final StockpileClient stockpile;
    private final CompletableFuture<Void> doneFuture = new CompletableFuture<>();
    private volatile boolean complete;
    private final Executor executor;
    private final ActorWithFutureRunner actor;

    private final Metrics queueMetrics = new Metrics("stockpile.write.queue");
    private final Metrics writeMetrics = new Metrics("stockpile.written");
    private final AtomicReference<CompletableFuture<Void>> notFull = new AtomicReference<>(new CompletableFuture<>());
    private final AtomicInteger limit = new AtomicInteger(1000);
    private final AtomicInteger size = new AtomicInteger();

    public StockpileShardWriter(int shardId, StockpileClient stockpile, Executor executor) {
        this.shardId = shardId;
        this.stockpile = stockpile;
        this.executor = executor;
        this.actor = new ActorWithFutureRunner(this::act, executor);
    }

    public CompletableFuture<Void> write(Metric metric) {
        try {
            if (doneFuture.isCompletedExceptionally()) {
                return doneFuture;
            }

            queueMetrics.plus(metric.archive);
            int queueSize = size.getAndIncrement();
            queue.add(metric);
            if (queueSize >= limit.get()) {
                CompletableFuture<Void> prevSync, next;
                do {
                    prevSync = notFull.get();
                    if (!prevSync.isDone()) {
                        next = prevSync;
                        break;
                    }
                    next = new CompletableFuture<>();
                } while (!notFull.compareAndSet(prevSync, next));

                actor.schedule();
                return next;
            } else {
                actor.schedule();
                return completedFuture(null);
            }
        } catch (Throwable e) {
            return failedFuture(e);
        }
    }

    public void complete() {
        complete = true;
        actor.schedule();
    }

    public CompletableFuture<Void> doneFuture() {
        return doneFuture;
    }

    private CompletableFuture<Void> act() {
        if (doneFuture.isDone()) {
            notFullNow();
            return completedFuture(null);
        }

        try {
            var commands = TShardCommandRequest.newBuilder().setShardId(shardId);
            Stats stats = new Stats();
            Metric metric;
            while ((metric = queue.poll()) != null) {
                size.decrementAndGet();
                notFullNow();

                var request = prepare(metric);
                stats.add(metric.archive);
                stats.bytes += request.getSerializedSize();
                commands.addCommands(request);
                if (stats.bytes >= BYTE_SIZE_LIMIT) {
                    return writeToStorage(commands.build(), stats);
                }

                var next = queue.peek();
                if (next == null) {
                    break;
                }

                if (stats.bytes + next.archive.bytesCount() > BYTE_SIZE_LIMIT) {
                    return writeToStorage(commands.build(), stats);
                }
            }

            if (commands.getCommandsCount() > 0) {
                return writeToStorage(commands.build(), stats);
            }

            if (!complete) {
                notFullNow();
                limit.incrementAndGet();
                return completedFuture(null);
            }

            if (queue.isEmpty()) {
                System.out.println("Complete write to shard " + StockpileShardId.toString(shardId));
                doneFuture.complete(null);
            }
        } catch (Throwable e) {
            doneFuture.completeExceptionally(e);
        }
        return completedFuture(null);
    }

    private CompletableFuture<Void> writeToStorage(TShardCommandRequest commands, Stats stats) {
        int size = commands.getSerializedSize();
        Supplier<CompletableFuture<Void>> body = () -> CompletableFutures.safeCall(() -> {
            return stockpile.bulkShardCommand(commands);
        }).thenAccept(response -> {
                    if (response.getStatus() != EStockpileStatusCode.OK) {
                        throw new StockpileRuntimeException(response.getStatus(), "shardId " + shardId + " for "+ DataSize.prettyString(size) + " " + response.getStatusMessage());
                    }
                });

        return RetryCompletableFuture.runWithRetries(body, RETRY_CONFIG)
            .whenComplete((r, e) -> {
                if (e != null) {
                    doneFuture.completeExceptionally(e);
                }
                writeMetrics.plus(stats);
                queueMetrics.minus(stats);
                actor.schedule();
            });
    }

    private TCommandRequest prepare(Metric metric) {
        return TCommandRequest.newBuilder()
            .setCompressedWrite(TCompressedWriteRequest.newBuilder()
                .setMetricId(MetricId.newBuilder()
                    .setShardId(metric.shardId)
                    .setLocalId(metric.localId)
                    .build())
                .addChunks(serialize(metric.archive))
                .setBinaryVersion(metric.archive.getFormat().getFormat())
                .build())
            .build();
    }

    private TimeSeries.Chunk serialize(MetricArchiveImmutable archive) {
        if (archive.hasColumn(StockpileColumn.MERGE) || archive.bytesCount() > ARCHIVE_BYTES_SIZE_TO_FORCE_DECIM) {
            archive = repackWithoutMerge(archive);
        }

        ByteString data = MetricArchiveNakedSerializer.serializerForFormatSealed(archive.getFormat())
            .serializeToByteString(archive);

        return TimeSeries.Chunk.newBuilder()
            .setContent(data)
            .setPointCount(archive.getRecordCount())
            .build();
    }

    private MetricArchiveImmutable repackWithoutMerge(MetricArchiveImmutable source) {
        MetricArchiveMutable archive = new MetricArchiveMutable(source.header());
        archive.ensureBytesCapacity(source.columnSetMask(), source.bytesCount());
        var point = RecyclableAggrPoint.newInstance();

        var it = source.iterator();
        if (source.bytesCount() > ARCHIVE_BYTES_SIZE_TO_FORCE_DECIM) {
            var decimPolicy = source.getOwnerProjectIdOrUnknown() == EProjectId.GOLOVAN
                ? EDecimPolicy.POLICY_5_MIN_AFTER_8_DAYS
                : EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS;

            it = DecimatingAggrGraphDataIterator.of(
                source.getType(),
                source.iterator(),
                DecimPoliciesPredefined.policyFromProto(decimPolicy),
                System.currentTimeMillis());
        }

        while (it.next(point)) {
            point.merge = false;
            archive.addRecordData(source.columnSetMask(), point);
        }
        point.recycle();
        return archive.toImmutableNoCopy();
    }

    private void notFullNow() {
        var sync = notFull.get();
        if (!sync.isDone()) {
            sync.completeAsync(() -> null, executor);
        }
    }

    private static class Stats {
        private long bytes;
        private long metrics;
        private long records;

        public void add(MetricArchiveImmutable archive) {
            this.metrics += 1;
            this.records += archive.getRecordCount();
        }
    }

    private static class Metrics {
        final GaugeInt64 bytes;
        final GaugeInt64 metrics;
        final GaugeInt64 records;

        Metrics(String prefix) {
            var registry = MetricRegistry.root();
            bytes = registry.gaugeInt64(prefix + ".bytes");
            metrics = registry.gaugeInt64(prefix + ".metrics");
            records = registry.gaugeInt64(prefix + ".records");
        }

        void plus(Stats stats) {
            bytes.add(stats.bytes);
            metrics.add(stats.metrics);
            records.add(stats.records);
        }

        void plus(MetricArchiveImmutable archive) {
            bytes.add(archive.bytesCount());
            records.add(archive.getRecordCount());
            metrics.add(1);
        }

        void minus(Stats stats) {
            bytes.add(-stats.bytes);
            metrics.add(-stats.metrics);
            records.add(-stats.records);
        }
    }
}
