package ru.yandex.solomon.tool.stockpile;

import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.stream.Stream;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.stockpile.client.StockpileClient;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileShardWriters {
    private final Executor executor;
    private final StockpileClient client;
    private volatile StockpileShardWriter[] writers;

    public StockpileShardWriters(StockpileClient stockpile, Executor executor) {
        stockpile.forceUpdateClusterMetaData().join();
        this.client = stockpile;
        this.executor = executor;
        this.writers = new StockpileShardWriter[0];
    }

    private void ensureWriterInit(int shardId) {
        var copy = writers;
        if (shardId >= copy.length) {
            int totalShards = Math.max(client.getTotalShardsCount(), shardId);
            if (copy.length < totalShards) {
                synchronized (this) {
                    writers = changeQueueSize(writers, totalShards);
                }
            }
        }
    }

    private StockpileShardWriter[] changeQueueSize(StockpileShardWriter[] prev, int expectedSize) {
        if (prev.length >= expectedSize) {
            return prev;
        }

        var result = Arrays.copyOf(prev, expectedSize);
        for (int index = prev.length; index < result.length; index++) {
            int shardId = index + 1;
            result[index] = new StockpileShardWriter(shardId, client, executor);
        }
        return result;
    }

    private StockpileShardWriter getWriter(int shardId) {
        ensureWriterInit(shardId);
        return writers[shardId - 1];
    }

    public CompletableFuture<Void> write(Metric metric) {
        try {
            ensureWriterInit(metric.shardId);
            return getWriter(metric.shardId).write(metric);
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    public CompletableFuture<Void> write(int shardId, long localId, MetricArchiveImmutable archive) {
        return write(new Metric(shardId, localId, archive));
    }

    public void complete() {
        for (var writer : writers) {
            writer.complete();
        }
    }

    public CompletableFuture<Void> doneFuture() {
        return Stream.of(writers)
            .map(StockpileShardWriter::doneFuture)
            .collect(collectingAndThen(toList(), CompletableFutures::allOfVoid));
    }
}
