package ru.yandex.solomon.tool.stockpile;

import java.util.ArrayList;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvClientSync;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.time.TimeUtils;
import ru.yandex.solomon.tool.stockpile.backup.BackupsHelper;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.kikimrKv.ShardIdMapToLong;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;
import ru.yandex.stockpile.server.data.names.InBackupFileNameSet;


/**
 * @author Stepan Koltsov
 */
@Component
@ParametersAreNonnullByDefault
public class StockpileToolCommon {

    private final KikimrKvClient kvClient;
    private final KikimrKvClientSync kvClientSync;
    private final BackupsHelper backupsHelper;

    @Autowired
    public StockpileToolCommon(KikimrKvClient kvClient) {
        this.kvClient = kvClient;
        this.kvClientSync = new KikimrKvClientSync(kvClient);
        this.backupsHelper = new BackupsHelper(kvClient);
    }

    public long[] resolveKvTabletsForShards(String path) {
        return CompletableFutures.join(resolveKvTabletsForShardsImpl(path));
    }

    private CompletableFuture<long[]> resolveKvTabletsForShardsImpl(String path) {
        return kvClient.resolveKvTablets(path);
    }

    @FunctionalInterface
    interface ShardCallback {
        CompletableFuture<Void> forShard(long tabletId);
    }

    void stockpileForEachShard(long[] tabletIds, ShardCallback shardCallback) {
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<>();

        for (long tabletId : tabletIds) {
            futures.add(shardCallback.forShard(tabletId));
        }

        CompletableFuture<Void> all = CompletableFutures.allOfVoid(futures);

        for (;;) {
            int remaining = Math.toIntExact(futures.stream().filter(f -> !f.isDone()).count());
            if (remaining == 0) {
                break;
            }

            System.out.println(remaining + " to go");
            try {
                all.get(500, TimeUnit.MILLISECONDS);
            } catch (Exception ignored) {
            }
        }

        all.join();

        System.out.println(tabletIds.length + " shards done");
    }


    public interface TempSnapshotCallback {
        void run(long tabletId, FileNamePrefix prefix, InBackupFileNameSet inBackupFileNameSet);
    }

    public void withTempSnapshot2(String kvVolumePath, StockpileShardId shardId, FileNamePrefix what, TempSnapshotCallback callback) {
        var shards = new ShardIdMapToLong(kvClient.resolveKvTablets(kvVolumePath).join());
        long tabletId = shards.get(shardId.getId());

        if (!what.format().equals(FileNamePrefix.Current.instance.format())) {
            // no need to create temp backup for non-current backup
            callback.run(tabletId, what, backupsHelper.listFilesInBackupParsed(tabletId, what));
        } else {
            FileNamePrefix.TempBackup tempBackupPrefix =
                new FileNamePrefix.TempBackup(new FileNamePrefix.Backup(TimeUtils.unixTime()));

            backupsHelper.backupSync(tabletId, 0, what, tempBackupPrefix);

            try {
                callback.run(tabletId, tempBackupPrefix, backupsHelper.listFilesInBackupParsed(tabletId, tempBackupPrefix));
            } finally {
                try {
                    backupsHelper.deleteBackups(tabletId, 0, tempBackupPrefix);
                } catch (Throwable e) {
                    e.printStackTrace();
                }
            }
        }
    }
}
