package ru.yandex.solomon.tool.stockpile.backup;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.StringMicroUtils;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.time.TimeUtils;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;
import ru.yandex.stockpile.server.data.names.InBackupFileNameSet;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;


/**
 * @author Sergey Polovko
 */
public class BackupsHelper {

    private final KikimrKvClient kvClient;

    public BackupsHelper(KikimrKvClient kvClient) {
        this.kvClient = kvClient;
    }

    public CompletableFuture<Void> backup(
        long tabletId, long tabletGen,
        FileNamePrefix what, FileNamePrefix.ExceptCurrent target)
    {
        return kvClient.copyRange(tabletId, tabletGen, NameRange.all(), target.format(), what.format(), 0);
    }

    public CompletableFuture<Void> backupCurrent(
        long tabletId, long tabletGen,
        FileNamePrefix.ExceptCurrent target)
    {
        return backup(tabletId, tabletGen, FileNamePrefix.Current.instance, target);
    }

    public void backupSync(long tabletId, long tabletGen, FileNamePrefix what, FileNamePrefix.ExceptCurrent backup) {
        backup(tabletId, tabletGen, what, backup).join();
    }

    public CompletableFuture<List<BackupWithStats>> listBackupsWithFiles(
        long tabletId, long tabletGen)
    {
        return kvClient.readRangeNames(tabletId, tabletGen, StringMicroUtils.asciiPrefixToRange(StockpileKvNames.BACKUP_FIRST_LETTER), 0)
            .thenApply(s -> s.stream()
                .collect(Collectors.groupingBy(
                    e -> FileNamePrefix.Backup.pf.parsePrefix(e.getName()).success().prefix(),
                    LinkedHashMap::new,
                    Collectors.toList()))
                .entrySet()
                .stream()
                .map(e -> new BackupWithStats(e.getKey(), e.getValue()))
                .collect(Collectors.toList()));
    }

    public LinkedHashMap<FileNamePrefix.Backup, List<BackupWithStats>> listBackups(long[] kvTablets) {
        ArrayList<CompletableFuture<List<BackupWithStats>>> futures = new ArrayList<>(kvTablets.length);
        for (long kvTablet : kvTablets) {
            futures.add(listBackupsWithFiles(kvTablet, 0)
                .exceptionally(e -> {
                    throw new RuntimeException("tabletId: " + kvTablet);
                }));
        }

        List<List<BackupWithStats>> backups = CompletableFutures.joinAll(futures);

        return backups.stream()
            .flatMap(Collection::stream)
            .collect(Collectors.groupingBy(BackupWithStats::getBackup, LinkedHashMap::new, Collectors.toList()));
    }

    public List<KikimrKvClient.KvEntryStats> listFilesInBackupSync(long tabletId, long tabletGen, FileNamePrefix prefix) {
        return listFilesInBackupAsync(tabletId, tabletGen, prefix).join();
    }

    public CompletableFuture<List<KikimrKvClient.KvEntryStats>> listFilesInBackupAsync(long tabletId, long tabletGen, FileNamePrefix prefix) {
        return kvClient.readRangeNames(tabletId, tabletGen, prefix.toNameRange(), 0);
    }

    @Nonnull
    public InBackupFileNameSet listFilesInBackupParsed(long tabletId, FileNamePrefix prefix) {
        List<KikimrKvClient.KvEntryStats> filesInBackup = listFilesInBackupSync(tabletId, 0, prefix);

        List<FileNameParsed> files = filesInBackup.stream()
            .map(e -> FileNameParsed.parseInBackup(prefix, e.getName()))
            .collect(Collectors.toList());

        return new InBackupFileNameSet(files);
    }

    public void deleteBackups(long tabletId, long tabletGen, FileNamePrefix.ExceptCurrent prefix) {
        deleteBackupsAsync(tabletId, tabletGen, prefix).join();
    }

    public CompletableFuture<Void> deleteBackupsAsync(long tabletId, long tabletGen, FileNamePrefix.ExceptCurrent prefix) {
        if (prefix.format().isEmpty()) {
            throw new IllegalArgumentException("empty prefix: " + prefix);
        }

        return kvClient.deleteRange(tabletId, tabletGen, prefix.toNameRange(), 0);
    }

    public CompletableFuture<Void> restoreBackup(long tabletId, FileNamePrefix.ExceptCurrent backup) {
        if (backup.format().isEmpty()) {
            return CompletableFuture.failedFuture(new IllegalArgumentException("empty prefix"));
        }

        return kvClient.incrementGeneration(tabletId, 0)
                .thenCompose(gen -> new BackupRestorePipeline(tabletId, gen, backup).run());
    }

    private final class BackupRestorePipeline {
        private final long tabletId;
        private final long tableGen;
        private final FileNamePrefix.ExceptCurrent backup;
        private final FileNamePrefix.TempBackup tempBackup;

        private BackupRestorePipeline(long tabletId, long tableGen, FileNamePrefix.ExceptCurrent backup) {
            this.tabletId = tabletId;
            this.tableGen = tableGen;
            this.backup = backup;
            this.tempBackup = new FileNamePrefix.TempBackup(new FileNamePrefix.Backup(TimeUtils.unixTime()));
        }

        public CompletableFuture<Void> run() {
            return ensureBackupExists()
                    .thenCompose(ignore -> makeTempBackup())
                    .thenCompose(ignore -> rmCurrentFiles())
                    .thenCompose(ignore -> restoreFromBackupFiles(backup))
                    .thenCompose(ignore -> rmBackup(backup))
                    .thenCompose(ignore -> rmBackup(tempBackup));
        }

        private CompletableFuture<Void> ensureBackupExists() {
            return kvClient.readRangeNames(tabletId, tableGen, backup.toNameRange(), 0)
                    .thenAccept(response -> {
                        if (response.isEmpty()) {
                            throw new IllegalStateException("Backup files absents: " + backup);
                        }
                    });
        }

        private CompletableFuture<Void> makeTempBackup() {
            return backupCurrent(tabletId, tableGen, tempBackup);
        }

        private CompletableFuture<Void> rmCurrentFiles() {
            return kvClient.deleteRange(tabletId, tableGen, StockpileKvNames.currentFilesRange(), 0);
        }

        private CompletableFuture<Void> restoreFromBackupFiles(FileNamePrefix.ExceptCurrent backup) {
            return kvClient.copyRange(tabletId, tableGen, backup.toNameRange(), FileNamePrefix.Current.instance.format(), backup.format(), 0);
        }

        private CompletableFuture<Void> rmBackup(FileNamePrefix.ExceptCurrent backup) {
            return kvClient.deleteRange(tabletId, tableGen, backup.toNameRange(), 0);
        }
    }
}
